# -*- coding: utf-8 -*-

from collections import namedtuple

from passport.backend.takeout.common.exceptions import (
    MalformedJobId,
    UnsupportedJobIdVersion,
)
from passport.backend.takeout.common.signatures import (
    check_signature,
    sign,
)


JobIdV1 = namedtuple('JobIdV1', ['service_name', 'uid', 'extract_id'])


# Не должен совпадать с разделителем у подписей
DELIMITER = '$'


def parse_job_id(job_id):
    if not job_id.startswith('v1' + DELIMITER):
        raise UnsupportedJobIdVersion('Unsupported job_id version')
    bits = job_id.split(DELIMITER)
    if len(bits) != 6:
        raise MalformedJobId('Malformed job_id')
    v1, signature, job_id_const, service_name, uid, extract_id = bits
    if job_id_const != 'job_id':
        raise MalformedJobId('Malformed job_id')
    check_signature([v1, job_id_const, service_name, uid, extract_id], signature)
    return JobIdV1(
        service_name=service_name,
        uid=uid,
        extract_id=extract_id,
    )


def make_job_id_v1(uid, service_name, extract_id):
    uid = str(uid)
    extract_id = str(extract_id)
    signature = sign(['v1', 'job_id', service_name, uid, extract_id])
    return DELIMITER.join([
        'v1',
        signature,
        'job_id',
        service_name,
        uid,
        extract_id,
    ])
