# -*- coding: utf-8 -*-
from __future__ import absolute_import

from contextlib import contextmanager
import logging

import redis
from redis import sentinel

from passport.backend.takeout.common.conf import db as conf_db
from passport.backend.takeout.common import conf


log = logging.getLogger('takeout.redis')


def get_redis():
    """
    :raises: redis.sentinel.MasterNotFoundError
    """
    config = conf.get_config()
    redis_conf = conf_db.get_redis_settings(config)
    sentinel = redis.Sentinel(
        redis_conf['sentinel'],
        socket_timeout=redis_conf['socket_timeout'],
        socket_connect_timeout=redis_conf['socket_connect_timeout'],
    )
    host, port = sentinel.discover_master(redis_conf['master_name'])
    return redis.Redis(
        host,
        port,
        socket_timeout=redis_conf['socket_timeout'],
        socket_connect_timeout=redis_conf['socket_connect_timeout'],
        password=redis_conf['password'],
    )


@contextmanager
def maybe_ignore_errors(ignore):
    try:
        yield
    except (sentinel.MasterNotFoundError, redis.RedisError) as e:
        log.warning('Redis error: %s %s', e.__class__.__name__, e)
        if not ignore:
            raise


def set_key_with_ttl(key, value, ttl, skip_errors=False):
    with maybe_ignore_errors(skip_errors):
        get_redis().setex(key, ttl, value)


def add_to_set(key, item, ttl, skip_errors=False):
    with maybe_ignore_errors(skip_errors):
        pipe = get_redis().pipeline()
        pipe.sadd(key, item)
        pipe.expire(key, ttl)
        pipe.execute()


def is_in_set(key, item, skip_errors=False):
    with maybe_ignore_errors(skip_errors):
        return get_redis().sismember(key, item)


def is_key_set(key, skip_errors=False):
    with maybe_ignore_errors(skip_errors):
        return get_redis().exists(key) > 0


def delete_key(key, skip_errors=False):
    with maybe_ignore_errors(skip_errors):
        get_redis().delete(key)
