# -*- coding: utf-8 -*-
import base64
import hashlib
import hmac
import json
import logging

from passport.backend.takeout.common import exceptions
from passport.backend.takeout.common.conf.signatures import (
    get_key_version,
    get_keys_for_signing,
)
from passport.backend.utils import time


DELIMITER = ':'


log = logging.getLogger('takeout.signatures')


def _serialize_data(data):
    return json.dumps(
        data,
        sort_keys=True,
    )


def sign(data, key_version=None, keys=None, salt=None):
    if salt is None:
        salt = str(time.get_unixtime())

    if key_version is None:
        key_version = str(get_key_version())
    if keys is None:
        keys = get_keys_for_signing()

    key = keys[key_version]

    log.debug('sign(%s, %r, %r)', data, key_version, salt)

    encoded_data = (salt + DELIMITER + _serialize_data(data)).encode('utf-8')

    digest = hmac.new(key, msg=encoded_data, digestmod=hashlib.sha256).digest()
    signature = base64.b64encode(digest).decode()
    log.debug('signature is %s', signature)
    return str(DELIMITER.join([salt, key_version, signature]))


def check_signature(signed_data, signature, keys=None):
    # принудительно приводим к строке, наблюдаем за эффектом
    signature = str(signature)

    if keys is None:
        keys = get_keys_for_signing()

    log.debug('check_signature(%r, %r)', signed_data, signature)

    signature_bits = signature.split(DELIMITER)
    if len(signature_bits) != 3:
        log.debug('Malformed signature')
        raise exceptions.MalformedSignature()
    salt, key_version, data = signature_bits
    if key_version not in keys:
        log.debug('Missing key by version %s', key_version)
        raise exceptions.MissingKeyByVersion()
    computed_signature = sign(signed_data, key_version, keys, salt=salt)

    if not hmac.compare_digest(computed_signature, signature):
        log.debug('Signature mismatch')
        raise exceptions.SignatureMismatch('Computed {} from {}, got {}'.format(
            repr(computed_signature),
            repr(signed_data),
            repr(signature),
        ))
