# -*- coding: utf-8 -*-
from passport.backend.core.s3 import get_s3_client
from passport.backend.takeout.common.conf import get_config
from passport.backend.takeout.common.constants import TOUCH_DIR_NAME
from passport.backend.takeout.common.redis import (
    delete_key,
    is_key_set,
    set_key_with_ttl,
)
from passport.backend.takeout.common.utils import (
    make_redis_touch_key,
    s3_path,
)


class TouchFiles(object):
    class Sync(object):
        # Синхронный процесс
        STARTED = 'sync-started'
        DONE = 'sync-done'

    class AsyncPoll(object):
        # Асинхронный процесс скачивания пользовательских данных в тейкаут начался
        STARTED = 'async-poll-started'
        DONE = 'async-poll-done'

    class AsyncUpload(object):
        # Отправили запрос смежному сервису о том, что пора готовить выгрузку данных
        ORDERED_DATA = 'async-upload-ordered-data'
        # Асинхронный процесс заливки пользовательских данных в тейкаут начался
        STARTED = 'async-upload-started'
        # Асинхронный процесс заливки пользовательских данных в тейкаут завершён
        DONE = 'async-upload-done'

    def __init__(self, uid, extract_id, service_name, s3_client=None):
        self.s3 = s3_client or get_s3_client()
        self.uid = uid
        self.extract_id = extract_id
        self.service_name = service_name
        self.folder = s3_path(self.uid, self.extract_id, self.service_name, TOUCH_DIR_NAME)

    @classmethod
    def from_job_id(cls, job_id, s3_client=None):
        return cls(job_id.uid, job_id.extract_id, job_id.service_name, s3_client=s3_client)

    def get_s3_key(self, touch_name):
        return s3_path(self.folder, touch_name)

    def get_redis_key(self, touch_name):
        return make_redis_touch_key(self.uid, self.extract_id, self.service_name, touch_name)

    def set(self, touch_name, skip_errors=False):
        """Создаёт запись в redis"""
        set_key_with_ttl(
            self.get_redis_key(touch_name),
            '',
            ttl=get_config()['redis']['touch_files_ttl'],
            skip_errors=skip_errors,
        )

    def unset(self, touch_name):
        """Удаляет запись из s3 и redis"""
        key = self.get_s3_key(touch_name)
        self.s3.delete_file(key=key)

        delete_key(self.get_redis_key(touch_name))

    def is_set(self, touch_name):
        """
        Проверяет наличие тач-файла в s3 или в redis
        :return: True если файл есть, False если нету
        """
        exists_in_s3 = self.s3.file_exists(key=self.get_s3_key(touch_name))
        exists_in_redis = is_key_set(self.get_redis_key(touch_name))

        return exists_in_s3 or exists_in_redis
