# -*- coding: utf-8 -*-
import errno
from io import BytesIO
import logging
import os
import os.path
import random
import re
import shutil
import string
import subprocess
import uuid

from passport.backend.core.s3 import (
    DELIMITER,
    get_s3_client,
    S3TemporaryError,
    UnseekableStreamError,
)
from passport.backend.takeout.common.conf import get_config
from passport.backend.takeout.common.conf.crypto import get_keys_for_encrypting
from passport.backend.takeout.common.constants import (
    STATUS_NO_DATA,
    STATUS_OK,
    TOUCH_DIR_NAME,
)
from passport.backend.takeout.common.crypto import (
    decrypt_stream,
    encrypt_stream,
)
from passport.backend.takeout.common.exceptions import ServicePermanentError
from passport.backend.takeout.common.redis import (
    add_to_set,
    is_in_set,
)
from passport.backend.utils.string import smart_bytes
from six import (
    iterkeys,
    string_types,
)


log = logging.getLogger('takeout.common.utils')


MAX_STDOUT_LENGTH = 1000

READ_BUFFER_SIZE = 1024 ** 2  # 1 мб, взял из головы

KEY_VERSION_REGEX = re.compile(r'\.kv(\d+)$')

S3_CHUNK_SIZE = 1000

RE_FILENAME = re.compile(r'^[a-z0-9._-]{1,255}$')


def s3_path(*args):
    return DELIMITER.join(map(str, args))


def s3_list_all_files(s3_client, folder, chunk_size=S3_CHUNK_SIZE):
    files = []
    for chunk in s3_client.iterate_files_by_chunks(folder=folder, chunk_size=chunk_size):
        files.extend(chunk)
    return files


def is_filename_valid_for_archive(filename):
    return bool(RE_FILENAME.match(filename))


def filename_and_key_version_to_s3_key(filename, key_version):
    if key_version is None:
        raise ValueError('key_version must not be None')
    return '%s.kv%s' % (filename, key_version)


def ignore_s3_file(key):
    bits = key.split(DELIMITER)
    if len(bits) > 2:
        return TOUCH_DIR_NAME == bits[-2]
    return False


def s3_key_to_filename_and_key_version(s3_key):
    search = KEY_VERSION_REGEX.search(s3_key)
    if not search:
        raise ValueError('Unable to find key version in \'%s\'' % s3_key)
    filename = s3_key[:search.start()]
    key_version = int(search.group(1))
    return filename, key_version


def encrypt_and_upload_to_s3(uid, extract_id, service_name, file_name, file_object, key_version=None):
    """
    Шифрует переданный файловый объект и загружает его в S3
    """
    keys = get_keys_for_encrypting()
    key_version = key_version or get_config()['s3']['encryption_key_version']
    s3 = get_s3_client()

    try:
        log.debug('Uploading %r', file_name)
        output_file = encrypt_stream(
            file_object,
            keys,
            key_version,
        )
        s3.upload_fileobj(
            file_object=output_file,
            dst_folder=DELIMITER.join([str(uid), str(extract_id), service_name]),
            dst_filename=filename_and_key_version_to_s3_key(file_name, key_version),
            # Клиент S3 не умеет за один проход выгружать файл и вычислять от
            # него хеш, поэтому для шифруемого потока придётся отключить
            # проверку целостности.
            skip_integrity_check=True,
        )
    except Exception as e:
        log.debug('Failed uploading %r with %r', file_name, e)
        if isinstance(e, UnseekableStreamError):
            # Шифрованный поток нельзя отмотать назад, поэтому его выгрузку
            # нельзя поретраить, и нужно ретраиться уровнем выше
            raise S3TemporaryError()
        raise
    else:
        log.debug('Successfully uploaded %r', file_name)


def make_redis_cache_key(uid, extract_id, service_name):
    return f'takeout:extract:{uid}:{extract_id}:{service_name}:visited_file_links'


def make_redis_touch_key(uid, extract_id, service_name, touch_name):
    return f'takeout:extract:touch:{uid}:{extract_id}:{service_name}:{touch_name}'


def download_file_links_and_encrypt_and_upload_to_s3(uid, extract_id, service_name, builder, builder_response,
                                                     use_redis_cache=False, key_version=None):
    if builder_response['status'] == STATUS_NO_DATA:
        pass
    elif builder_response['status'] == STATUS_OK:
        if builder_response.get('data'):
            for filename, content in builder_response['data'].items():
                if not is_filename_valid_for_archive(filename):
                    raise ServicePermanentError('Got unacceptable filename {}'.format(repr(filename)))
                if not isinstance(content, string_types):
                    raise ServicePermanentError('Content of file {} must be string-encoded'.format(repr(filename)))
                encrypt_and_upload_to_s3(
                    uid=uid,
                    extract_id=extract_id,
                    service_name=service_name,
                    file_name=filename,
                    file_object=BytesIO(smart_bytes(content)),
                    key_version=key_version,
                )
        if builder_response.get('file_links'):
            config = get_config()
            # надеюсь, что там будет список
            file_links_count = len(builder_response['file_links'])
            log.debug('Got %d file links', file_links_count)
            for i, file_link in enumerate(builder_response['file_links'], start=1):
                redis_key = make_redis_cache_key(uid, extract_id, service_name)
                if use_redis_cache and is_in_set(
                    key=redis_key,
                    item=file_link,
                    skip_errors=True,
                ):
                    log.debug('Skipping %r: it seems to have been downloaded earlier', file_link)
                    continue

                file_name, file_object = builder.get_info_from_url(file_link)
                encrypt_and_upload_to_s3(
                    uid=uid,
                    extract_id=extract_id,
                    service_name=service_name,
                    file_name=file_name,
                    file_object=file_object,
                    key_version=key_version,
                )

                if use_redis_cache:
                    add_to_set(
                        key=redis_key,
                        item=file_link,
                        ttl=config['redis']['visited_links_cache_ttl'],
                        skip_errors=True,
                    )

                log.debug('%s/%s file links processed', i, file_links_count)
    else:
        raise NotImplementedError('Unknown status: %s' % builder_response['status'])  # noqa


def s3_get_uploaded_files(uid, extract_id, service_name):
    """
    Возвращает уже присутствующие файлы для сервиса в выгрузке
    :return: Список имён файлов
    """
    s3 = get_s3_client()

    s3_folder = s3_path(uid, extract_id, service_name)

    files_info = s3_list_all_files(s3, s3_folder)

    uploaded_files = []
    for file_info in files_info:
        # удаляю суффикс с версией ключа для шифрования
        key = file_info['Key'].rsplit('.', 1)[0]
        # игнорирую технические файлы
        if ignore_s3_file(file_info['Key']):
            continue
        filename = os.path.basename(key)

        uploaded_files.append(filename)

    return uploaded_files


def check_files_existance(files_list, path):
    """
    Проверка списка файлов на то, существуют ли они или нет
    :param files_list: список путей
    :param path: путь на файловой системе
    :return: список пар (file, True или False)
    """
    path = os.path.expanduser(path)
    retval = []
    for file_name in files_list:
        file_on_fs = os.path.join(path, file_name)
        retval.append((file_name, os.path.exists(file_on_fs)))
    return retval


def maybe_make_dirs(path):
    try:
        os.makedirs(path)
    except OSError as e:
        if e.errno == errno.EEXIST:
            return
        raise


def download_from_s3_and_decrypt(uid, extract_id, destination_folder):
    """
    Выкачивает из S3 все файлы из заданного каталога, дешифрует и сохраняет результат локально.
    """
    keys = get_keys_for_encrypting()
    s3 = get_s3_client()

    s3_folder = '{}/{}/'.format(uid, extract_id)

    files_info = s3_list_all_files(s3, s3_folder)
    # размер нужен для передачи его в функцию дешифрации
    file_to_size = {f['Key']: f['Size'] for f in files_info if not ignore_s3_file(f['Key'])}

    files = [
        f
        for f, exists in check_files_existance(iterkeys(file_to_size), destination_folder)
        if not exists
    ]

    for file_key in files:
        downloading_stream = BytesIO()
        s3.download_fileobj(file_key, downloading_stream)

        filename, key_version = s3_key_to_filename_and_key_version(file_key)

        # Отрезаем от имени файла префикс s3_folder: мы хотим в destination_folder получить не дерево папок,
        # а непосредственно содержимое каталога s3_folder в S3
        destination_file_path = os.path.join(destination_folder, filename.replace(s3_folder, ''))

        maybe_make_dirs(os.path.dirname(destination_file_path))

        downloading_stream.seek(0)

        with open(destination_file_path, 'bw') as f:
            decrypted_file = decrypt_stream(
                downloading_stream,
                keys,
                key_version,
                file_to_size[file_key],
            )
            while True:
                data = decrypted_file.read(READ_BUFFER_SIZE)
                if data:
                    f.write(data)
                else:
                    break


def rename_output_folders_if_needed(data_folder):
    conf_services = get_config()['services'].items()

    for service_name, service_dict in conf_services:
        output_folder = service_dict.get('folder')
        if output_folder is not None and output_folder != service_name:
            service_folder = os.path.join(data_folder, service_name)

            if os.path.exists(service_folder):
                new_service_folder = os.path.join(data_folder, output_folder)
                maybe_make_dirs(new_service_folder)

                for file_to_move in os.listdir(service_folder):
                    shutil.move(
                        os.path.join(service_folder, file_to_move),
                        os.path.join(new_service_folder, file_to_move),
                    )

                shutil.rmtree(service_folder)


def make_archive_unsafe(data_folder, archive_folder, archive_name, archive_password):
    # 7z определяет тип архива по расширению, а мы всегда хотим zip-формат
    tmp_archive_name = archive_name + '.zip'

    try:
        subprocess.check_output(
            [
                '7z',
                'a',  # добавить файлы в архив
                '-mx=1',  # уровень сжатия - минимальный (самый быстрый)
                '-mem=AES128',  # метод шифрования
                '-mtc=off',  # не сохраняем времена модификации файлов
                '-p%s' % archive_password,  # пароль к архиву
                tmp_archive_name,
                '%s/*' % data_folder,
            ],
            stderr=subprocess.STDOUT,
            cwd=archive_folder,
        )
        tmp_archive_filepath = os.path.join(archive_folder, tmp_archive_name)
        dst_archive_filepath = os.path.join(archive_folder, archive_name)
        shutil.move(tmp_archive_filepath, dst_archive_filepath)
        return dst_archive_filepath
    except subprocess.CalledProcessError as e:
        raise RuntimeError(
            'Error while creating archive (returncode %s): %s' % (e.returncode, e.output[-MAX_STDOUT_LENGTH:]),
        )


def make_password(length):
    alphabet = string.ascii_uppercase + string.ascii_lowercase + string.digits
    return ''.join(random.sample(alphabet, length))


def request_id_pop(request_id):
    # удаляет последний элемент из request_id
    return ','.join(map(str, request_id.split(',')[:-1]))


def get_task_id():
    return str(uuid.uuid4())[-8:]


def get_extract_id():
    return uuid.uuid4().hex
