# -*- coding: utf-8 -*-
import argparse
import sys

from passport.backend.core.s3 import get_s3_client
from passport.backend.takeout.common.conf import configure_settings
from passport.backend.takeout.common.utils import s3_list_all_files


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        'action',
        metavar='ACTION',
        nargs=1,
        choices=[
            'list',
            'list_all',
            'delete',
            'upload',
        ],
        help='Desired action',
    )
    parser.add_argument('--folder')
    parser.add_argument('--key')
    args = parser.parse_args()
    return args


def print_files(file_infos):
    print('\t'.join(['File name'.ljust(100), 'File size'.rjust(10), 'Modification time']))
    for file_info in file_infos:
        print('\t'.join([
            file_info['Key'].ljust(100),
            str(file_info['Size']).rjust(10),
            str(file_info['LastModified'].replace(tzinfo=None)),
        ]))


def run_app():
    configure_settings()
    s3 = get_s3_client()

    args = parse_args()
    action = args.action[0]

    if action == 'list':
        files = s3.list_files(folder=args.folder, limit=1000, last_seen_key=args.key)
        print_files(files)
    elif action == 'list_all':
        files = s3_list_all_files(s3, folder=args.folder)
        print_files(files)
    elif action == 'delete':
        if not args.folder and not args.key:
            print('You should specify --folder or --key')
            sys.exit(1)
        if args.folder:
            for file_info in s3.list_files(folder=args.folder):
                s3.delete_file(key=file_info['Key'])
        elif args.key:
            s3.delete_file(key=args.key)
    elif action == 'upload':
        if not args.key:
            print('You should specify --key')
            sys.exit(1)
        folder, filename = args.key.rsplit('/', 1)
        s3.upload_fileobj(sys.stdin, folder, filename, skip_integrity_check=True)
