# -*- coding: utf-8 -*-
import argparse
from collections import namedtuple
from datetime import datetime
import itertools

from passport.backend.core.s3 import get_s3_client
from passport.backend.takeout.common.conf import configure_settings
from passport.backend.takeout.common.touch import (
    TOUCH_DIR_NAME,
    TouchFiles,
)
from passport.backend.takeout.common.utils import s3_list_all_files


FileInfo = namedtuple('FileInfo', ['key', 'size', 'mtime'])


DATETIME_FORMATS = [
    '%Y-%m-%d %H:%M:%S',
    '%Y-%m-%d %H:%M',
    '%Y-%m-%d %H',
    '%Y-%m-%d',
    '%Y-%m',
]


def _parse_datetime(value):
    caught_e = None
    for datetime_format in DATETIME_FORMATS:
        try:
            return datetime.strptime(value, datetime_format)
        except ValueError as e:
            caught_e = e
    raise argparse.ArgumentTypeError(str(caught_e))


def parse_date_range(value):
    if not value:
        return None
    if '..' in value:
        if value.count('.') > 2:
            raise argparse.ArgumentTypeError('Only two dots are allowed')
        left, right = value.split('..')
        return _parse_datetime(left), _parse_datetime(right)
    return [_parse_datetime(value), _parse_datetime(value).replace(hour=23, minute=59, second=59)]


def str2bool(value):
    if value.lower() in ('yes', 'true', 't', 'y', '1'):
        return True
    elif value.lower() in ('no', 'false', 'f', 'n', '0'):
        return False
    else:
        raise argparse.ArgumentTypeError('Boolean value expected')


def print_extract_info(files, date_range, print_touch, print_normal, only_unfinished, only_finished, short):
    files = sorted(files, key=lambda f: f.key)

    for key, group in itertools.groupby(files, key=lambda f: f.key.split('/')[:2]):
        uid, extract_id = key
        group = list(group)
        start_time = min(group, key=lambda f: f.mtime).mtime
        if not short:
            print('=' * 40)
            print('UID: {}\tExtract id: {}'.format(uid, extract_id))
            print('Start time: {}'.format(start_time.strftime('%Y-%m-%d %H:%M:%S')))

        for service_name, service_files in itertools.groupby(group, key=lambda f: f.key.split('/')[2]):
            print_this_service = False
            is_finished = False

            touch_files = []
            extracted_files = []

            for f in service_files:
                if date_range is None or (date_range[0] <= f.mtime <= date_range[1]):
                    f_key_bits = f.key.split('/')

                    if f_key_bits[3] == TOUCH_DIR_NAME:
                        touch_files.append((f_key_bits[4], f.mtime))
                    else:
                        extracted_files.append((f_key_bits[3], f.mtime))

            for touch_file, _ in touch_files:
                if touch_file in [
                    TouchFiles.Sync.DONE,
                    TouchFiles.AsyncUpload.DONE,
                    TouchFiles.AsyncPoll.DONE,
                ]:
                    is_finished = True
                    break

            if only_unfinished and not is_finished:
                print_this_service = True

            if only_finished and is_finished:
                print_this_service = True

            if not only_finished and not only_unfinished:
                print_this_service = True

            if print_this_service:
                if not short:
                    print('  {}:'.format(service_name))

                if print_touch:
                    for (file_name, mtime) in touch_files:
                        if short:
                            print('{}/{}/{}\t{}\t{}'.format(
                                uid,
                                extract_id,
                                service_name,
                                file_name.upper(),
                                mtime.strftime('%Y-%m-%d %H:%M:%S'),
                            ))
                        else:
                            print('    {}\t{}'.format(file_name.upper().ljust(30), mtime.strftime('%Y-%m-%d %H:%M:%S')))

                if print_normal:
                    for (file_name, mtime) in extracted_files:
                        if short:
                            print('{}/{}/{}\t{}\t{}'.format(
                                uid,
                                extract_id,
                                service_name,
                                file_name,
                                mtime.strftime('%Y-%m-%d %H:%M:%S'),
                            ))
                        else:
                            print('    {}\t{}'.format(file_name.ljust(30), mtime.strftime('%Y-%m-%d %H:%M:%S')))


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('prefixes', nargs='*')
    parser.add_argument('--date', '-d', default=None, required=False, type=parse_date_range)
    parser.add_argument('--touch', '-t', default=True, required=False, type=str2bool)
    parser.add_argument('--normal', '-n', default=True, required=False, type=str2bool)
    parser.add_argument('--only-unfinished', '-ou', default=False, required=False, action='store_true')
    parser.add_argument('--only-finished', '-of', default=False, required=False, action='store_true')
    parser.add_argument('--short', '-s', default=False, required=False, action='store_true')
    args = parser.parse_args()
    return args


def run_app():
    configure_settings()

    args = parse_args()

    s3 = get_s3_client()

    prefixes = args.prefixes or ['']
    for prefix in prefixes:
        files = [FileInfo(f['Key'], f['Size'], f['LastModified'].replace(tzinfo=None)) for f in s3_list_all_files(s3, folder=prefix)]

        print_extract_info(
            files,
            date_range=args.date,
            print_touch=args.touch,
            print_normal=args.normal,
            only_unfinished=args.only_unfinished,
            only_finished=args.only_finished,
            short=args.short,
        )
