# -*- coding: utf-8 -*-
import argparse
import collections
import json
import os
import sys

import yaml


Config = collections.namedtuple('Config', ['keyring_testing', 'keyring_production'])


def panic(*text):
    sys.stderr.write(' '.join(map(str, text)) + '\n')
    sys.exit(1)


def read_config(path):
    with open(path) as f:
        config = yaml.safe_load(f)
        return Config(**config)


def check_client_id_and_aliases_are_unique(keyring_config, alias, client_id):
    alias_count = sum(1 for d in keyring_config['destinations'] if d['alias'] == alias)
    client_id_count = sum(1 for d in keyring_config['destinations'] if d['client_id'] == client_id)
    # client_id нельзя переопределять, поэтому если нашли такой же, но от другого алиаса, кидаем ошибку
    if client_id_count > 0:
        if [d for d in keyring_config['destinations'] if d['client_id'] == client_id and d['alias'] != alias]:
            panic('Client id {} is set to {} alias'.format(client_id, alias))
    # алиас можно переопределять, поэтому одно вхождение это ок, просто заменится
    if alias_count > 1:
        panic('Alias {} is set to client id {}'.format(alias, client_id))


def add_to_keyring(filename, alias, client_id):
    with open(filename) as f:
        keyring = json.load(f)
    check_client_id_and_aliases_are_unique(keyring, alias, client_id)
    for dst_config in keyring['destinations']:
        if dst_config['alias'] == alias:
            dst_config['client_id'] = client_id
            break
    else:
        keyring['destinations'].append({
            'client_id': client_id,
            'alias': alias,
        })
    keyring['destinations'].sort(key=lambda dst_config: dst_config['alias'])
    with open(filename, 'w') as f:
        f.write(
            json.dumps(
                keyring,
                sort_keys=True,
                indent=2,
            ) + '\n',
        )


def del_from_keyring(filename, alias):
    with open(filename) as f:
        keyring = json.load(f)
    keyring['destinations'] = [dst_config for dst_config in keyring['destinations'] if dst_config['alias'] != alias]
    with open(filename, 'w') as f:
        f.write(
            json.dumps(
                keyring,
                sort_keys=True,
                indent=2,
            ) + '\n',
        )


def run_add(args):
    config = read_config(os.path.abspath(args.config))
    if args.client_id_testing is not None:
        add_to_keyring(config.keyring_testing, args.alias, args.client_id_testing)
    if args.client_id_production is not None:
        add_to_keyring(config.keyring_production, args.alias, args.client_id_production)


def run_del(args):
    if not (args.test or args.prod):
        build_parser().error('No environment provided, add --test or --prod')

    config = read_config(os.path.abspath(args.config))
    if args.test:
        del_from_keyring(config.keyring_testing, args.alias)
    if args.prod:
        del_from_keyring(config.keyring_production, args.alias)


def run_list(args):
    config = read_config(os.path.abspath(args.config))
    keyring_test = dict((c['alias'], c['client_id']) for c in json.load(open(config.keyring_testing))['destinations'])
    keyring_prod = dict((c['alias'], c['client_id']) for c in json.load(open(config.keyring_production))['destinations'])
    aliases = set(keyring_prod.keys()) | set(keyring_test.keys())
    if not aliases:
        return

    longest_alias_length = len(max(aliases, key=len))

    for alias in sorted(aliases):
        print('{}\t{}\t{}'.format(
            alias.rjust(longest_alias_length) if sys.stdout.isatty() else alias,
            keyring_test.get(alias, '-'),
            keyring_prod.get(alias, '-'),
        ))


def nullable_int(x):
    if str(x).strip() == '-':
        return None
    else:
        return int(x)


def build_parser():
    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers()

    add_subparser = subparsers.add_parser('add')
    add_subparser.add_argument('alias')
    add_subparser.add_argument('client_id_testing', type=nullable_int)
    add_subparser.add_argument('client_id_production', type=nullable_int)
    add_subparser.add_argument('-c', '--config', default=os.path.abspath(
        os.path.join(
            os.path.curdir,
            'tvmkm.conf',
        ),
    ))
    add_subparser.set_defaults(func=run_add)

    del_subparser = subparsers.add_parser('del')
    del_subparser.add_argument('alias')
    del_subparser.add_argument('-t', '--test', action='store_true', help='Delete from test keyring')
    del_subparser.add_argument('-p', '--prod', action='store_true', help='Delete from production keyring')
    del_subparser.add_argument('-c', '--config', default=os.path.abspath(
        os.path.join(
            os.path.curdir,
            'tvmkm.conf',
        ),
    ))
    del_subparser.set_defaults(func=run_del)

    list_subparser = subparsers.add_parser('list')
    list_subparser.add_argument('-c', '--config', default=os.path.abspath(
        os.path.join(
            os.path.curdir,
            'tvmkm.conf',
        ),
    ))
    list_subparser.set_defaults(func=run_list)

    return parser


def parse_args():
    parser = build_parser()
    return parser.parse_args()


def run_app():
    args = parse_args()
    args.func(args)
