# -*- coding: utf-8 -*-

import logging

from passport.backend.core.logbroker.exceptions import BaseLogbrokerError
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.s3 import S3TemporaryError
from passport.backend.logbroker_client.core.handlers.exceptions import HandlerException
from passport.backend.takeout.common.exceptions import (
    ResponseNotReadyError,
    RetryLater,
    ServiceTemporaryError,
)


KNOWN_EXCEPTIONS = (
    BaseLogbrokerError,
    S3TemporaryError,
    ServiceTemporaryError,
)

log = logging.getLogger('takeout.tasks')


class LogbrokerWriterError(HandlerException):
    """ Не удалось отправить сообщение с помощью logbroker_writer """


def is_known_exception(e):
    for known_e in KNOWN_EXCEPTIONS:
        if isinstance(e, known_e):
            return True
    return False


def process_exception(e, uid, extract_id, task_name, statbox_params, task_id=None, retries=0, max_retries=0):
    statbox_params.update(
        task_id=task_id,
        uid=uid,
        extract_id=extract_id,
        task_name=task_name,
        log_source='logbroker',
        tskv_format='takeout-log',
    )

    statbox_logger = StatboxLogger(**statbox_params)

    statbox_logger.bind(
        error=e.__class__.__name__,
        error_message=trim_message(str(e)),
        retries=retries,
    )

    can_be_retried = True

    if isinstance(e, (RetryLater, ResponseNotReadyError)):
        log.debug('Task %s was delayed: %s', task_name, e.args[0] if len(e.args) else type(e).__name__)
        statbox_logger.bind(status='retry_later')
    else:
        if max_retries == 0 or retries < max_retries:
            log_message = 'Task %s failed (%s: %s) and will be retried'
            log_args = (
                log_message,
                task_name,
                e.__class__.__name__,
                trim_message(str(e)),
            )
            statbox_logger.bind(status='retry_failure')
        else:
            log_message = 'Task %s failed (%s: %s) and will not be retried anymore'
            log_args = (
                log_message,
                task_name,
                e.__class__.__name__,
                trim_message(str(e)),
            )
            statbox_logger.bind(status='failure')
            can_be_retried = False

        if is_known_exception(e):
            log.warning(*log_args)
        else:
            log.exception(*log_args)

    statbox_logger.log()
    return can_be_retried
