# -*- coding: utf-8 -*-
import logging
import time

from passport.backend.core.protobuf.takeout.takeout_pb2 import TakeoutRequest
from passport.backend.takeout.common.job_id import make_job_id_v1
from passport.backend.takeout.common.logbroker import get_takeout_logbroker_writer
from passport.backend.takeout.logbroker_client.exception_handler import process_exception
from passport.backend.takeout.logbroker_client.tasks.async_task import (
    async_task_get,
    async_task_start,
)
from passport.backend.takeout.logbroker_client.tasks.async_upload_task import (
    async_upload_task_get,
    async_upload_task_start,
)
from passport.backend.takeout.logbroker_client.tasks.sync_task import sync_task


log = logging.getLogger('takeout.tasks.extract_task')


def send_extract_service_task_message(
    source, task_id, uid, extract_id, unixtime, service_name, service_enabled=True, step=None,
    retries=0, max_retries=0, job_id=None, require_enabled=True, delay_until=0
):
    if require_enabled and not service_enabled:
        return True

    task_base_message = TakeoutRequest.TaskBaseMessage(
        task_id=task_id,
        uid=uid,
        extract_id=extract_id,
        unixtime=unixtime,
        task_name='extract_service',
        delay_until=delay_until,
        retries=retries,
        max_retries=max_retries,
        source=source,
        send_time=int(time.time()),
    )
    task_detail_message = TakeoutRequest.TaskExtractServiceMessage(
        service=service_name,
        step=step,
        job_id=job_id,
    )

    try:
        logbroker = get_takeout_logbroker_writer('takeout_tasks')

        logbroker.send(
            TakeoutRequest(task_base_message=task_base_message, extract_service=task_detail_message)
        )
        return True
    except Exception as e:
        statbox_params = {
            'service_name': service_name,
            'archive_requested_unixtime': unixtime,
        }
        process_exception(
            task_id=task_id,
            uid=uid,
            extract_id=extract_id,
            task_name='send_extract_service',
            statbox_params=statbox_params,
            e=e,
            retries=retries,
            max_retries=max_retries,
        )
        return False


def sync_extract_service_task(uid, extract_id, service_name, unixtime):
    sync_task(
        uid=uid,
        extract_id=extract_id,
        service_name=service_name,
        unixtime=unixtime,
    )


def async_extract_service_start_task(uid, extract_id, service_name, unixtime):
    job_id = async_task_start(
        uid=uid,
        extract_id=extract_id,
        service_name=service_name,
        unixtime=unixtime,
    )

    return job_id


def async_extract_service_get_task(uid, extract_id, service_name, job_id):
    async_task_get(
        uid=uid,
        extract_id=extract_id,
        service_name=service_name,
        job_id=job_id,
    )


def async_upload_service_start_task(uid, extract_id, service_name, unixtime):
    job_id = make_job_id_v1(uid, service_name, extract_id)

    async_upload_task_start(
        uid=uid,
        extract_id=extract_id,
        service_name=service_name,
        unixtime=unixtime,
        job_id=job_id,
        statbox=None,
    )

    return job_id


def async_upload_service_get_task(uid, extract_id, service_name):
    async_upload_task_get(
        uid=uid,
        extract_id=extract_id,
        service_name=service_name,
    )
