# -*- coding: utf-8 -*-
import logging
import time

from passport.backend.core.protobuf.takeout.takeout_pb2 import TakeoutRequest
from passport.backend.takeout.common.conf import get_config
from passport.backend.takeout.common.conf.services import get_service_configs
from passport.backend.takeout.common.exceptions import ResponseNotReadyError
from passport.backend.takeout.common.logbroker import get_takeout_logbroker_writer
from passport.backend.takeout.common.touch import TouchFiles
from passport.backend.takeout.logbroker_client.exception_handler import process_exception
from passport.backend.takeout.logbroker_client.tasks.make_archive import make_archive


log = logging.getLogger('takeout.tasks.make_archive')


def send_make_archive(
    source,
    task_id,
    uid,
    extract_id,
    services,
    unixtime,
    retries=0,
    delay_until=0,
    manual_takeout=None,
):
    task_base_message = TakeoutRequest.TaskBaseMessage(
        delay_until=delay_until,
        extract_id=extract_id,
        retries=retries,
        task_id=task_id,
        task_name='make_archive',
        uid=uid,
        unixtime=unixtime,
        source=source,
        send_time=int(time.time()),
    )
    task_detail_message = TakeoutRequest.TaskMakeArchiveMessage(
        is_manual_takeout=bool(manual_takeout),
        services=services,
    )

    try:
        logbroker = get_takeout_logbroker_writer('takeout_tasks')

        logbroker.send(
            TakeoutRequest(task_base_message=task_base_message, make_archive=task_detail_message)
        )
        return True
    except Exception as e:
        statbox_params = {
            'services': services,
            'archive_requested_unixtime': unixtime,
        }
        process_exception(
            task_id=task_id,
            uid=uid,
            extract_id=extract_id,
            task_name='make_archive',
            statbox_params=statbox_params,
            e=e,
        )
        return False


def check_touch_files_done(uid, extract_id, services, retries):
    service_configs = get_service_configs(services=services)
    need_full_check = retries > get_config()['check_touch_files']['without_full_check']
    not_ready_services = []

    for service_name in services:
        touch = TouchFiles(uid, extract_id, service_name)
        service_type = service_configs[service_name].type
        if (
            service_type == 'sync' and not touch.is_set(TouchFiles.Sync.DONE) or
            service_type == 'async' and not touch.is_set(TouchFiles.AsyncPoll.DONE) or
            service_type == 'async_upload' and not touch.is_set(TouchFiles.AsyncUpload.DONE)
        ):
            if need_full_check:
                not_ready_services.append(service_name)
            else:
                raise ResponseNotReadyError('%s not ready' % service_name)

    if len(not_ready_services) > 0:
        raise ResponseNotReadyError('Services not ready: %s' % ','.join(not_ready_services))

    return True


def make_archive_task(uid, extract_id, unixtime, cooking_directory, manual_takeout=None):
    make_archive(
        cooking_directory=cooking_directory,
        extract_id=extract_id,
        manual_takeout=manual_takeout,
        uid=uid,
        unixtime=unixtime,
    )
