# -*- coding: utf-8 -*-
import logging
import os
import shutil
import time

from passport.backend.core.builders.blackbox import get_blackbox
from passport.backend.core.builders.passport import (
    Passport,
    PassportAccountNotFoundError,
    PassportActionNotRequiredError,
)
from passport.backend.core.s3 import (
    DELIMITER,
    get_s3_client,
)
from passport.backend.core.utils.blackbox import get_many_accounts_by_uids
from passport.backend.takeout.common.conf import get_config
from passport.backend.takeout.common.utils import (
    download_from_s3_and_decrypt,
    make_archive_unsafe,
    make_password,
    maybe_make_dirs,
    rename_output_folders_if_needed,
)


log = logging.getLogger('takeout.tasks.make_archive')


def make_archive(uid, extract_id, unixtime, cooking_directory, manual_takeout=None):
    blackbox = get_blackbox()
    accounts, unknown_uids = get_many_accounts_by_uids(
        [uid],
        blackbox,
        dict(
            need_public_name=False,
            attributes=['takeout.archive_s3_key'],
        ),
    )
    if unknown_uids:
        log.debug('Uid {} doesn\'t exist. Skipping'.format(uid))
        return
    account = accounts[0]
    if account.takeout.archive_s3_key and extract_id in account.takeout.archive_s3_key:
        log.debug('Archive already exists. Skipping')
        return

    s3 = get_s3_client()
    config = get_config()

    # не используем tempfile, чтобы ретраи таски использовали один и тот же workdir
    working_folder = os.path.join(
        cooking_directory,
        str(uid),
        str(extract_id),
    )
    data_folder = os.path.join(working_folder, '_data')
    maybe_make_dirs(data_folder)

    # Выкачиваем все собранные данные
    ts = time.time()
    download_from_s3_and_decrypt(uid, extract_id, destination_folder=data_folder)
    log.debug('Downloaded from S3 in {:.4f}'.format(time.time() - ts))

    rename_output_folders_if_needed(data_folder)

    archive_filename = 'archive_%s_%s_%s.7z' % (extract_id, uid, int(unixtime))
    archive_filepath = os.path.join(working_folder, archive_filename)
    password_filepath = '%s.pwd' % archive_filepath

    # Генерим пароль для архива
    if not os.path.exists(password_filepath):
        archive_password = make_password(length=config['archive']['password_length'])
        with open(password_filepath, 'w') as f:
            f.write('%s\n' % archive_password)
    else:
        with open(password_filepath, 'r') as f:
            archive_password = f.read().strip()

    # Собираем архив
    if not os.path.exists(archive_filepath):
        ts = time.time()
        make_archive_unsafe(
            data_folder=data_folder,
            archive_folder=working_folder,
            archive_name=archive_filename,
            archive_password=archive_password,
        )
        log.debug('Made archive in {:.4f}'.format(time.time() - ts))

    # Заливаем архив в S3
    s3_folder = DELIMITER.join(['res', str(uid), str(extract_id)])
    archive_s3_key = DELIMITER.join([s3_folder, archive_filename])
    ts = time.time()
    s3.upload_file(
        local_filepath=archive_filepath,
        dst_folder=s3_folder,
        dst_filename=archive_filename,
    )
    log.debug('Uploaded to S3 in {:.4f}'.format(time.time() - ts))

    if not manual_takeout:
        # Сообщаем паспорту о готовности архива
        try:
            Passport(use_tvm=config['passport']['use_tvm']).takeout_finish_extract(
                uid=uid,
                archive_s3_key=archive_s3_key,
                archive_password=archive_password,
            )
        except (PassportActionNotRequiredError, PassportAccountNotFoundError) as e:
            log.debug('Ignoring error from passport: {}'.format(e.__class__.__name__))

    # Удаляем временные файлы
    shutil.rmtree(working_folder, ignore_errors=True)

    log.debug('Done! S3 key: %s', archive_s3_key)
