from itertools import chain
import re

from ci.tasklet.common.proto import service_pb2
from passport.backend.tasklets.changelog.proto import schema_tasklet


REVERT_REGEX = re.compile(r'Revert \"(.*)\"')


class ChangelogImpl(schema_tasklet.ChangelogBase):
    """
    Форк https://a.yandex-team.ru/arc_vcs/ci/tasklet/registry/demo/changelog/
    """

    LIMIT = 300

    def run(self):
        request = service_pb2.GetCommitsRequest()
        request.flow_launch_id = self.input.context.job_instance_id.flow_launch_id
        request.limit = ChangelogImpl.LIMIT

        self.output.changelog.version = (
            self._build_version_from_special_version()
            if self.input.config.special_version
            else self._build_version_from_prefix()
        )

        commit_message_to_issues = {}
        response = self.ctx.ci.GetCommits(request)
        for commit in response.commits:
            message = self._format(commit.message)
            commit_message_to_issues[message] = commit.issues

        if response.HasField('next'):
            # TODO: можно передать значение поля next из ответа в следующий запрос (поле offset)
            #  для получения следующей пачки коммитов
            commit_message_to_issues['...truncated...'] = []

        for message in reversed(list(commit_message_to_issues.keys())):  # от самого старого к самому новому
            revert_info = REVERT_REGEX.search(message)
            # Если данный коммит - реверт другого коммита из этого же релиза, то уберём оба из ченжлога
            if revert_info:
                original_message = self._format(revert_info.group(1))
                if original_message in commit_message_to_issues:
                    commit_message_to_issues.pop(original_message)
                    commit_message_to_issues.pop(message)

        changelog = reversed(list(commit_message_to_issues.keys()))  # хотим самые старые коммиты видеть первыми
        issues = sorted(set(
            chain.from_iterable(commit_message_to_issues.values()),
        ))

        self.output.changelog.content.extend(changelog)
        self.output.changelog.issues.extend(issues)

    @staticmethod
    def _format(commit_message):
        # Отрезается лишнее (например, описание пулл-реквеста)
        message = commit_message.rsplit('\n')[0]
        # Удаляем артефакты релиз-машины и CI
        for regex in (
            r'\[mergeto:[\w-]+:\d+\]',
            r'\(cherry-pick from [\w-]+\)',
        ):
            message = re.sub(regex, '', message)
        return message.strip()

    def _build_version_from_prefix(self):
        return '.'.join(
            map(
                str,
                [
                    self.input.config.version_prefix,
                    self.input.context.version_info.major,
                    self.input.context.version_info.minor or 0,
                ],
            ),
        )

    def _build_version_from_special_version(self):
        return '%s-%s.%s' % (
            self.input.config.special_version,
            self.input.context.version_info.major,
            self.input.context.version_info.minor or 0,
        )
