from datetime import (
    datetime,
    timedelta,
)
import json
import logging
from time import sleep

from ci.tasklet.common.proto import service_pb2 as ci
from passport.backend.clients.conductor import ConductorClient
from passport.backend.tasklets.conductor.proto import wait_for_tasklet
from tasklet.services.yav.proto import yav_pb2


logger = logging.getLogger(__name__)


class WaitForConductorTicketImpl(wait_for_tasklet.WaitForConductorTicketBase):
    @staticmethod
    def _wait(start_datetime):
        seconds_passed = (datetime.now() - start_datetime).total_seconds()
        if seconds_passed < 10:
            time_to_sleep = 1
        elif seconds_passed < 300:
            time_to_sleep = 10
        elif seconds_passed < 3600:
            time_to_sleep = 30
        elif seconds_passed < 86400:
            time_to_sleep = 120
        else:
            time_to_sleep = 600

        sleep(time_to_sleep)

    def run(self):
        spec = yav_pb2.YavSecretSpec(uuid=self.input.context.secret_uid, key='conductor.token')
        oauth_token = self.ctx.yav.get_secret(spec).secret
        conductor_client = ConductorClient(token=oauth_token)

        progress_msg = ci.TaskletProgress()
        progress_msg.job_instance_id.CopyFrom(self.input.context.job_instance_id)
        progress_msg.module = 'CONDUCTOR'
        progress_msg.url = conductor_client.make_ticket_url(self.input.config.ticket_key)
        self.ctx.ci.UpdateProgress(progress_msg)

        status = None
        start_datetime = datetime.now()
        while datetime.now() - start_datetime < timedelta(seconds=self.input.config.time_to_wait):
            ticket_info = json.loads(conductor_client.get_ticket(self.input.config.ticket_key))
            status = ticket_info.get('value', {}).get('status', '-')
            if status == 'done':
                progress_msg.text = 'Тикет выложен'
                progress_msg.status = ci.TaskletProgress.Status.SUCCESSFUL
                self.ctx.ci.UpdateProgress(progress_msg)
                return
            elif status in ['obsolete', 'failed', 'revoked']:
                progress_msg.text = f'Выкладка сломалась (статус: {status})'
                progress_msg.status = ci.TaskletProgress.Status.FAILED
                self.ctx.ci.UpdateProgress(progress_msg)
                raise RuntimeError(progress_msg.text)
            else:
                progress_msg.text = f'Тикет выкладывается (текущий статус: {status})'
                self.ctx.ci.UpdateProgress(progress_msg)

            self._wait(start_datetime=start_datetime)

        progress_msg.text = f'Не дождались окончания выкладки (текущий статус: {status})'
        progress_msg.status = ci.TaskletProgress.Status.FAILED
        self.ctx.ci.UpdateProgress(progress_msg)
        raise RuntimeError(progress_msg.text)
