import logging
import re

from ci.tasklet.common.proto import service_pb2 as ci
from passport.backend.tasklets.startrek.proto import create_release_tasklet
from startrek_client import Startrek
from tasklet.services.yav.proto import yav_pb2


logger = logging.getLogger(__name__)


COMMON_ST_TAG = 'CI'
CHANGELOG_ISSUE_REGEX = re.compile(r'^\* .+$', re.MULTILINE)


class CreateReleaseIssueImpl(create_release_tasklet.CreateReleaseIssueBase):
    @staticmethod
    def merge_changelog_lines(old_lines, new_lines):
        result = old_lines
        for line in new_lines:
            if line not in old_lines:
                result.append(line)
        return result

    def make_ticket_description(self, changelog_lines):
        changelog = '\n'.join(changelog_lines)
        return f'(({self.input.context.ci_job_url} CI))\n\n{changelog}'

    def run(self):
        progress_msg = ci.TaskletProgress()
        progress_msg.job_instance_id.CopyFrom(self.input.context.job_instance_id)
        progress_msg.module = 'STARTREK'
        self.ctx.ci.UpdateProgress(progress_msg)

        spec = yav_pb2.YavSecretSpec(uuid=self.input.context.secret_uid, key='startrek.token')
        oauth_token = self.ctx.yav.get_secret(spec).secret
        startrek_client = Startrek(
            useragent='Passport CI',
            base_url='https://st-api.yandex-team.ru',
            token=oauth_token,
        )

        queue = self.input.issue.queue or 'PASSP'
        changelog_lines = [f'* {line}' for line in self.input.issue.changelog_lines]
        version = self.input.issue.package_version
        version_without_patch, _, _ = version.rpartition('.')

        common_issue_kwargs = dict(
            assignee=self.input.context.flow_triggered_by,
            summary=f'Релиз {self.input.issue.package_name} {version}',
            components=list(self.input.issue.components or []),
        )

        tags = [COMMON_ST_TAG, self.input.issue.package_name, version_without_patch]
        possible_issues = startrek_client.issues.find(
            query=' AND '.join(
                [
                    f'Queue: "{queue}"',
                ] + [
                    f'Tags: "{tag}"'
                    for tag in tags
                ],
            ),
            order=['-updated'],
        )

        if possible_issues:
            release_issue = possible_issues[0]
            changelog_lines_from_issue = CHANGELOG_ISSUE_REGEX.findall(release_issue.description)
            changelog_lines = self.merge_changelog_lines(
                old_lines=changelog_lines_from_issue,
                new_lines=changelog_lines,
            )
            release_issue.update(
                description=self.make_ticket_description(changelog_lines),
                **common_issue_kwargs,
            )

            if release_issue.status.key != 'open':
                release_issue.transitions['reopen'].execute()
        else:
            release_issue = startrek_client.issues.create(
                queue=queue,
                createdBy=self.input.context.flow_triggered_by,
                type={'name': 'Release'},
                tags=tags,
                description=self.make_ticket_description(changelog_lines),
                **common_issue_kwargs,
            )

        issue_key = release_issue.key

        self.output.result.key = issue_key
        self.output.result.url = 'https://st.yandex-team.ru/%s' % issue_key

        progress_msg.url = self.output.result.url
        progress_msg.text = 'Тикет'
        progress_msg.status = ci.TaskletProgress.Status.SUCCESSFUL
        self.ctx.ci.UpdateProgress(progress_msg)
