import logging
from urllib.parse import quote

from ci.tasklet.common.proto import service_pb2 as ci
from passport.backend.tasklets.startrek.proto import transit_tasklet
from startrek_client import (
    exceptions,
    Startrek,
)
from tasklet.services.yav.proto import yav_pb2


logger = logging.getLogger(__name__)


ALLOWED_QUEUES = (
    'OAUTHREG',
    'PASSP',
    'PASSPADMIN',
    'PASSPINFRA',
    'PASSPORTDUTY',
    'VAULT',
)


class TransitIssuesImpl(transit_tasklet.TransitIssuesBase):
    @staticmethod
    def filter_issues(issues):
        return [
            issue
            for issue in issues
            if any(
                issue.startswith(f'{queue}-')
                for queue in ALLOWED_QUEUES
            )
        ]

    @staticmethod
    def get_issues_url(issues):
        query = ' or '.join([f'Key:{issue}' for issue in issues])
        return 'https://st.yandex-team.ru/issues/?_q={}'.format(quote(query))

    def run(self):
        issues = self.filter_issues(self.input.config.issues)
        if not issues:
            logger.info('Nothing to do')
            return

        progress_msg = ci.TaskletProgress()
        progress_msg.job_instance_id.CopyFrom(self.input.context.job_instance_id)
        progress_msg.module = 'STARTREK'
        progress_msg.url = self.get_issues_url(issues)
        progress_msg.text = 'Список тикетов'
        progress_msg.progress = 0
        self.ctx.ci.UpdateProgress(progress_msg)

        spec = yav_pb2.YavSecretSpec(uuid=self.input.context.secret_uid, key='startrek.token')
        oauth_token = self.ctx.yav.get_secret(spec).secret
        startrek_client = Startrek(
            useragent='Passport CI',
            base_url='https://st-api.yandex-team.ru',
            token=oauth_token,
        )

        total = len(issues)
        for current_num, issue in enumerate(issues):
            transitions = startrek_client.issues[issue].transitions.get_all()
            for transition in transitions:
                if transition.to.key == self.input.config.status:
                    try:
                        transition.execute(
                            resolution=self.input.config.resolution or None,
                        )
                        break
                    except exceptions.NotFound:
                        logger.warning(
                            f'Unable to transit {issue} to status {self.input.config.status}: transition `{transition.id}` is not allowed',
                        )
            else:
                logger.info(
                    f'Unable to transit {issue} to status {self.input.config.status}: no applicable transition found. Available transitions are {transitions}',
                )
            progress_msg.progress = current_num / total
            self.ctx.ci.UpdateProgress(progress_msg)

        progress_msg.status = ci.TaskletProgress.Status.SUCCESSFUL
        self.ctx.ci.UpdateProgress(progress_msg)
