# -*- coding: utf-8 -*-

from __future__ import print_function

import argparse
import base64
import collections
import datetime
import json
import logging
import sys

from library.python.vault_client.instances import Production as VaultClient
from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.social.common.db.utils import (
    build_master_db_config,
    build_slave_db_config,
    create_engine,
)
from passport.backend.social.common.dev_server.dev_server import append_python_path2
from passport.backend.social.common.importer import (
    install_file_system_importer,
    install_python_path2,
)
from passport.backend.social.common.provider_settings import providers
from passport.backend.social.common.providers.Apple import Apple
from passport.backend.social.common.session import Session
from passport.backend.social.common.social_config import social_config


DST_SECRET = 'sec-01dcrxvq4z7vexgxaevc4js9vq'
KEY_ID_LENGTH = 10
META_LENGTH = 21
SRC_SECRET = 'sec-01g6trpabnj37ygmd4ackyb6g8'


logger = logging.getLogger(__name__)
yav = None


def main():
    logging.basicConfig(level=logging.INFO)

    args_parser = ArgsParser()
    args = args_parser.parse_args()

    if args.update_apps:
        return sys.exit(update_apps(args))
    elif args.list_key_ids:
        return sys.exit(list_key_ids(args))
    else:
        return sys.exit(update_secrets(args))


def update_secrets(args):
    global yav

    yav = VaultClient(decode_files=True)
    src_secret = yav.get_version(SRC_SECRET)
    validate_secret(src_secret)

    apple_keys = AppleKeys.from_secret_value(src_secret['value'])
    update_secret(apple_keys)

    apple_apps = parse_apple_apps(src_secret['value'])
    print(format_apple_apps(apple_apps))


def validate_secret(secret):
    for key in secret['value']:
        if len(key) in (KEY_ID_LENGTH, META_LENGTH):
            pass
        else:
            logger.warning('Unable to process key: ' + key)


def update_secret(apple_keys):
    dst_secret = list()
    for key_id, private_key in apple_keys.items():
        dst_secret.append(dict(
            key='APPLE_JWT_CERTIFICATE_' + key_id,
            value=base64.b64encode(private_key).decode(),
            encoding='base64',
        ))

    yav.create_diff_version(yav.get_version(DST_SECRET)['version'], dst_secret)


def parse_apple_apps(secret):
    apps = list()

    for key in secret:
        if len(key) == META_LENGTH:
            key_id = key[:KEY_ID_LENGTH]
            team_id, app_id = secret[key].split('.', 1)
            apps.append(AppleApp(app_id, key_id, team_id))

    return apps


def format_apple_apps(apple_apps):
    d = dict()
    for app in apple_apps:
        d[app.app_id] = dict(
            apple_jwt_certificate_id=app.key_id,
            apple_team_id=app.team_id,
        )
    return json.dumps(d, indent=4)


def update_apps(args):
    init_social()

    apps = json.load(sys.stdin)

    for app_id in apps:
        update_app(
            app_id,
            apps[app_id]['apple_jwt_certificate_id'],
            apps[app_id]['apple_team_id'],
        )


def update_app(app_id, key_id, team_id):
    app = providers.get_application_by_provider_app_id(Apple.id, app_id)
    if not app:
        logger.warning('Application not found: ' + app_id)
        return
    session = Session()
    session.add_committed(app)
    app.apple_jwt_certificate_id = key_id
    app.apple_team_id = team_id
    session.commit()


def list_key_ids(args):
    apps = json.load(sys.stdin)

    _get_key_id = lambda app_id: apps[app_id]['apple_jwt_certificate_id']

    print('# config')
    for app_id in sorted(apps, key=_get_key_id):
        print(format_key_id_for_config(apps[app_id]['apple_jwt_certificate_id']))

    print('\n# secret config')
    for app_id in sorted(apps, key=_get_key_id):
        print(format_key_id_for_secret_config(apps[app_id]['apple_jwt_certificate_id']))


def format_key_id_for_config(key_id):
    return 'apple_jwt_certificate_%(key_id)s = secrets.apple_jwt_certificate_%(key_id)s' % dict(key_id=key_id)


def format_key_id_for_secret_config(key_id):
    return "apple_jwt_certificate_%(key_id)s = b'''{{ APPLE_JWT_CERTIFICATE_%(key_id)s }}'''" % dict(key_id=key_id)


def init_social():
    append_python_path2('file::///usr/lib/yandex/socialism')

    install_python_path2()
    install_file_system_importer()

    social_config.init()
    providers.init()

    LazyLoader.register('chrono', lambda: datetime.datetime)
    LazyLoader.get_instance('chrono')

    read_conn = create_engine(build_slave_db_config(social_config), False)
    LazyLoader.register('slave_db_engine', lambda: read_conn)
    LazyLoader.get_instance('slave_db_engine')

    write_conn = create_engine(build_master_db_config(social_config), False)
    LazyLoader.register('master_db_engine', lambda: write_conn)
    LazyLoader.get_instance('master_db_engine')


class AppleKeys(object):
    def __init__(self):
        self._keys = dict()

    @classmethod
    def from_secret_value(cls, secret_value):
        self = cls()

        for key in secret_value:
            if len(key) == KEY_ID_LENGTH:
                self._keys[key] = secret_value[key]

        return self

    def items(self):
        return self._keys.items()


AppleApp = collections.namedtuple('AppleApp', 'app_id key_id team_id')


class ArgsParser(argparse.ArgumentParser):
    def __init__(self, *args, **kwargs):
        super(ArgsParser, self).__init__(*args, **kwargs)

        self.add_argument('-a', dest='update_apps', action='store_true', default=False)
        self.add_argument('-L', dest='list_key_ids', action='store_true', default=False)


if __name__ == '__main__':
    main()
