# -*- coding: utf-8 -*-

import datetime
import logging
import os

from passport.backend.tools.as_data_generator import (
    processes,
    settings,
)
from passport.backend.tools.as_data_generator.utils import (
    decompress_gzipped_file,
    download_file,
    get_changes_message,
    temporary_dir,
)
from passport.backend.tools.as_data_generator.validator import (
    preprocess_ipv6_origin,
    validate_as_name,
    validate_ip_origin,
)


log = logging.getLogger(__name__)


def generate_as_data(repo_url, username, usermail):
    with temporary_dir() as repo_path:
        log.info('Temporary folder: %s', repo_path)
        processes.git_clone(repo_url, repo_path)

        last_commit_datetime = processes.git_last_commit_datetime(repo_path)
        log.info('Last commit datetime: %s', last_commit_datetime)

        if last_commit_datetime > datetime.datetime.today() - datetime.timedelta(hours=1):
            log.info('Modification do not required')
            return

        updated_files = []
        for url, target_gz_name in settings.URL_TO_TARGET_FILE_NAME_MAPPING.items():
            target_gz_path = os.path.join(repo_path, target_gz_name)
            download_file(url, target_gz_path)
            target_path = target_gz_path.replace('.gz', '')
            decompress_gzipped_file(target_gz_path, target_path)
            log.info('Downloaded and decompressed: %s', target_path)

            path_to_validate = target_path
            if url == settings.IPV6_ORIGIN_URL:
                log.info('Preprocessing IPv6 origin file')
                path_to_validate = os.path.join(repo_path, settings.IPV6_PREPROCESSED_FILE_NAME)
                preprocess_ipv6_origin(target_path, path_to_validate)

            if processes.git_check_unstaged_changes(repo_path):
                log.info('Validating %s', path_to_validate)

                if url in (settings.IPV4_ORIGIN_URL, settings.IPV6_ORIGIN_URL):
                    validate_ip_origin(path_to_validate)
                else:
                    validate_as_name(path_to_validate)

                updated_files.append(os.path.basename(path_to_validate))

                processes.git_add(repo_path)

        if updated_files:
            message = get_changes_message(updated_files, repo_path)
            processes.dch(repo_path, username, usermail, message)
            processes.git_add(repo_path)
            processes.git_commit(
                repo_path,
                username,
                usermail,
                message,
            )
            processes.git_push(repo_path)
            log.info(message)
        else:
            log.info('There are no modified files in upstream')
