# -*- coding: utf-8 -*-
import datetime
from subprocess import (
    PIPE,
    Popen,
)

from passport.backend.tools.as_data_generator.exceptions import BasePassportAsDataGenerator


class ProcessError(BasePassportAsDataGenerator):
    pass


class GitError(ProcessError):
    pass


class GitCloneError(GitError):
    pass


class GitDiffError(GitError):
    pass


class GitAddError(GitError):
    pass


class GitPushError(GitError):
    pass


class GitCommitError(GitError):
    pass


class GitLastCommitUnixtimeError(GitError):
    pass


class DchError(ProcessError):
    pass


class SendmailError(ProcessError):
    pass


def _process_handling(process, exception_cls, ignore_err=True, return_exit_code=False):
    out, err = process.communicate()
    if (not ignore_err and err) or (not return_exit_code and process.returncode > 0):
        raise exception_cls('returncode: %s, err: %s, out: %s' % (process.returncode, err, out))
    if return_exit_code:
        return process.returncode
    return out


def git_clone(repo_url, repo_dir):
    process = Popen(['git', 'clone', repo_url, repo_dir], stdout=PIPE, stderr=PIPE, text=True)
    return _process_handling(process, GitCloneError)


def git_check_unstaged_changes(repo_dir):
    process = Popen(['git', 'diff', '--quiet'], stdout=PIPE, stderr=PIPE, cwd=repo_dir, text=True)
    return _process_handling(process, GitDiffError, return_exit_code=True)


def git_cached_changes_stat(repo_dir):
    process = Popen(['git', 'diff', '--numstat', '--cached'], stdout=PIPE, stderr=PIPE, cwd=repo_dir, text=True)
    return _process_handling(process, GitDiffError)


def git_add(repo_dir):
    process = Popen(['git', 'add', '-u'], stdout=PIPE, stderr=PIPE, cwd=repo_dir, text=True)
    return _process_handling(process, GitAddError)


def git_commit(repo_dir, username, usermail, message):
    process = Popen(
        ['git', 'commit', '-m', message, '--author="%s <%s>"' % (username, usermail)],
        cwd=repo_dir,
        stdout=PIPE,
        stderr=PIPE,
        text=True,
    )
    return _process_handling(process, GitCommitError)


def git_last_commit_datetime(repo_dir):
    process = Popen(
        ['git', 'show', '-s', '--format=%ct', 'HEAD'],
        cwd=repo_dir,
        stdout=PIPE,
        stderr=PIPE,
        text=True,
    )
    last_commit_unixtime = _process_handling(process, GitLastCommitUnixtimeError)
    return datetime.datetime.fromtimestamp(int(last_commit_unixtime))


def git_push(repo_dir):
    process = Popen(['git', 'push'], stdout=PIPE, stderr=PIPE, cwd=repo_dir, text=True)
    return _process_handling(process, GitPushError, ignore_err=True)


def dch(repository_dir, username, usermail, message):
    process = Popen(
        ['dch', '-i', message, '--distributor=debian', '--no-auto-nmu'],
        stdout=PIPE,
        stderr=PIPE,
        cwd=repository_dir,
        env={
            'DEBFULLNAME': username,
            'DEBEMAIL': usermail,
        },
        text=True,
    )
    return _process_handling(process, DchError)


def sendmail(message):
    process = Popen(['/usr/sbin/sendmail', '-t'], stdin=PIPE, stdout=PIPE, stderr=PIPE, text=True)
    process.stdin.write(message)
    return _process_handling(process, SendmailError)
