# -*- coding: utf-8 -*-
from contextlib import contextmanager
from email.mime.text import MIMEText
import gzip
import shutil
import socket
import struct
import tempfile

from passport.backend.tools.as_data_generator import settings
from passport.backend.tools.as_data_generator.exceptions import InvalidIpAddress
from passport.backend.tools.as_data_generator.processes import (
    git_cached_changes_stat,
    sendmail,
)
import requests


def decompress_gzipped_file(gzipped_path, target_path):
    with open(target_path, 'wb') as target_file, gzip.open(gzipped_path, 'rb') as gzipped_file:
        target_file.writelines(gzipped_file)


def download_file(url, target_path):
    r = requests.get(url, stream=True)
    with open(target_path, 'wb') as f:
        for chunk in r.iter_content(chunk_size=4096):
            if chunk:
                f.write(chunk)
                f.flush()


@contextmanager
def temporary_dir():
    tmp_dir = tempfile.mkdtemp()
    try:
        yield tmp_dir
    finally:
        shutil.rmtree(tmp_dir)


def ipv6_to_int(ipv6_addr):
    try:
        ipv6_addr = socket.inet_pton(socket.AF_INET6, ipv6_addr)
        high, low = struct.unpack(">QQ", ipv6_addr)
        return (high << 64) | low
    except (socket.error, struct.error) as e:
        raise InvalidIpAddress('Failed to convert IPv6 to integer: %s' % e)


def send_email(text, subject='Passport AS data generator error', recipients=settings.EMAIL_RECIPIENTS):
    msg = MIMEText(text.encode('utf-8'), _subtype='text/plain', _charset='utf-8')
    msg['Subject'] = subject
    msg['To'] = ', '.join(recipients)
    sendmail(msg.as_string())


def get_changes_message(updated_files, repo_path):
    changes_stat = [change_line.split() for change_line in git_cached_changes_stat(repo_path).strip().split('\n')]
    changes_stat = dict([(name, (added, removed)) for (added, removed, name) in changes_stat])

    return 'Updated files: ' + ', '.join(
        ['%s (added %s, removed %s)' % (name, changes_stat[name][0], changes_stat[name][1]) for name in updated_files],
    )
