# -*- coding: utf-8 -*-

from passport.backend.tools.as_data_generator.exceptions import (
    InvalidAsNameFile,
    InvalidIpAddress,
    InvalidIpOriginFile,
)
from passport.backend.tools.as_data_generator.utils import ipv6_to_int


def preprocess_ipv6_origin(source_path, target_path):
    with open(source_path, 'r') as source_file, open(target_path, 'w') as target_file:
        for line_num, line in enumerate(source_file):
            splitted = line.strip('\n').split()
            if len(splitted) < 2:
                raise InvalidIpOriginFile('%s:%s: not enough data in line' % (source_path, line_num))
            try:
                ip0, ip1 = map(ipv6_to_int, splitted[0].split('-'))
            except (ValueError, InvalidIpAddress) as e:
                raise InvalidIpOriginFile('%s:%s: %s' % (source_path, line_num, e))
            as_list = splitted[1:]
            target_file.write('\t'.join([str(ip0), str(ip1)] + as_list) + '\n')


def validate_ip_origin(path):
    with open(path, 'r') as source_file:
        if not source_file:
            raise InvalidIpOriginFile('%s: no data in file' % path)

        prev_ip1 = None
        for line_num, line in enumerate(source_file):
            splitted = line.strip('\n').split('\t')
            if len(splitted) < 3:
                raise InvalidIpOriginFile('%s:%s: not enough data in line' % (path, line_num))
            try:
                ip0, ip1 = map(int, splitted[0:2])
            except ValueError as e:
                raise InvalidIpOriginFile('%s:%s: %s' % (path, line_num, e))
            if ip0 > ip1:
                raise InvalidIpOriginFile('%s:%s: invalid IP range' % (path, line_num))

            if prev_ip1 is not None and ip0 <= prev_ip1:
                raise InvalidIpOriginFile('%s:%s: unsorted or overlapping IP ranges' % (path, line_num))
            prev_ip1 = ip1


def validate_as_name(path):
    with open(path, 'rb') as source_file:
        for line_num, line in enumerate(source_file):
            try:
                as_name, as_alias, as_description = line.decode('utf8', 'replace').split('\t')
            except ValueError as e:
                raise InvalidAsNameFile('%s:%s: %s' % (path, line_num, e))
