import collections
import itertools
import json
import operator
import argparse
import requests
import re
import sys
import io
import startrek_client


MACRO_CACHE = dict()
P_ID_CACHE = dict()
ABC_INFO = dict()


NetworkInfo = collections.namedtuple('NetworkInfo', ['network', 'owners', 'macro'])


SKIP_KEY_LEN = len('https://st-api.yandex-team.ru/v2/issues/')


class UnknownProjectID(Exception):
    pass


def run_app():
    parser = argparse.ArgumentParser()
    parser.add_argument("grants_file", metavar="FILENAME")
    parser.add_argument("--session-id", metavar="SESSION_ID", required=True)
    parser.add_argument('--phpsessid', metavar='PHPSESSID', required=True)
    parser.add_argument('--token', metavar='OAUTH_TOKEN', required=True)
    parser.add_argument('--find-consumers', action='store_true', default=False)
    parser.add_argument('--ticket', action='store_true', default=False)

    args = parser.parse_args()

    if args.find_consumers:
        build_info(args)
    if args.ticket:
        create_tasks(args)


def create_tasks(args):
    client = startrek_client.Startrek(
        useragent='passport-grants-review',
        base_url='https://st-api.yandex-team.ru/',
        token=args.token,
    )
    for task in open('tasks'):
        task = json.loads(task)
        issue = client.issues.create(
            queue='PASSPORTGRANTS',
            type={'name': 'Task'},
            components=['#GrantsReview'],
            summary=task['summary'],
            description=task['text'],
            followers=task['owners'],
        )
        print('https://st.yandex-team.ru/%s' % issue.self[SKIP_KEY_LEN:])


def build_info(args):
    with open(args.grants_file) as f:
        all_grants = json.load(f)

    f_out = open('tasks', 'w')

    for i, (consumer, consumer_data) in enumerate(sorted(all_grants.items(), key=operator.itemgetter(0))):
        out = io.StringIO()
        if consumer_data.get("client").get("client_id"):
            continue
        print("===Описание события", file=out)
        print("Привет!", file=out)
        print("В рамках подготовки сервиса Яндекс.Паспорт к аудиту мы запустили процесс пересмотра выданных прав потребителям внешнего (не корпоративного) инстанса.", file=out)
        print("\n", file=out)
        print(f"Для потребителя %%{consumer}%% был выдан доступ на использование методов и данных из паспорта.", file=out)
        print(f"Потребитель %%{consumer}%%", file=out)
        print("====+Гранты", file=out)
        print_grants(consumer_data["grants"], file=out)
        print("====+Сети", file=out)
        networks = []
        for network in consumer_data["networks"]:
            try:
                info = get_network_info(args.session_id, args.phpsessid, args.token, network)
            except UnknownProjectID:
                networks.append(
                    NetworkInfo(network, [], "Без Project ID")
                )
            else:
                networks.append(
                    NetworkInfo(network, info['owners'], info['macro'])
                )
        networks = sorted(networks, key=lambda x: x.macro)
        global_owners = set()
        for k, g in itertools.groupby(networks, key=lambda x: x.macro):
            print(f"=====%%{k}%%", file=out)
            g = list(g)
            owners = set()
            for item in g:
                owners |= set(item.owners)
            if owners:
                svcs = set()
                abc_admins = set()
                abc_managers = set()
                for o in list(owners):
                    if not o.startswith('svc_') and not o.startswith('yandex_'):
                        continue
                    abc_info = get_abc_info(args.token, o)
                    if abc_info:
                        abc_admins = set(abc_info['abc_admins'])
                        abc_managers = set(abc_info['abc_managers'])
                    else:
                        sys.stderr.write(f'Got nothing for service {o}\n')
                humans = set()
                robots = set()
                for o in sorted(owners | abc_admins | abc_managers):
                    if o.startswith('svc_') or o.startswith('yandex_'):
                        svcs.add(o)
                    elif o.startswith('robot-'):
                        robots.add(o)
                    else:
                        humans.add(o)
                        global_owners.add(o)
                if svcs:
                    print("Сервисы из racktables: " + ", ".join(sorted(svcs)), file=out)
                if humans | robots:
                    print("Владельцы из racktables или ABC: " + ", ".join(f"кто:{o}" for o in sorted(humans | robots)), file=out)
            for item in sorted(g, key=lambda x: x.network):
                print(f"- %%{item.network}%%", file=out)
            print('\n', file=out)

        if not global_owners:
            sys.stderr.write(f"Consumer {consumer} has no owners: {global_owners}\n")
        f_out.write('%s\n' % json.dumps(
            {
                'summary': f'Пересмотр грантов passport-api БЕЗ tvm: consumer={consumer}',
                'text': out.getvalue(),
                'owners': sorted(global_owners),
            },
        ))

    f_out.close()


def print_grants(grants, file):
    for key, data in sorted(grants.items(), key=operator.itemgetter(0)):
        if issubclass(type(data), collections.Mapping):
            for subkey, subdata in sorted(data.items(), key=operator.itemgetter(0)):
                for item in sorted(subdata):
                    print(f" - %%{key}.{subkey}.{item}%%", file=file)
        elif issubclass(type(data), collections.Sequence):
            for item in sorted(data):
                print(f" - %%{key}.{item}%%", file=file)
        else:
            raise ValueError(repr(data))


def guess_project_id(ip_address):
    if '@' in ip_address:
        return ip_address.split('@')[0]
    if ':' in ip_address:
        try:
            project_id = ip_address.split(':')[5]
            if project_id.startswith('/') or not project_id:
                pass
            else:
                return project_id
        except IndexError:
            pass
    raise UnknownProjectID(f'Project ID not found for {ip_address}')


def get_network_info(session_id, php_sessid, oauth_token, ip_address):
    project_id = guess_project_id(ip_address)

    macro_data = macros_by_project_id(session_id, php_sessid, project_id)
    try:
        macro_raw = macro_data['rows'][0]['macro']
    except IndexError:
        raise UnknownProjectID(f'Project ID not found for {ip_address}')
    macro = re.search(r"project_name=([a-zA-Z_0-9]+)'>", macro_raw).group(1)

    owners = []
    while not owners:
        info = get_firewall_info(oauth_token, macro)
        owners = info['owners']
        if owners:
            return {
                'owners': owners,
                'macro': macro,
            }
        if info['parent']:
            macro = info['parent']
        else:
            raise ValueError(':/')


def macros_by_project_id(session_id, php_sessid, project_id):
    if project_id in P_ID_CACHE:
        return P_ID_CACHE[project_id]

    rsp = requests.post(
        'https://racktables.yandex-team.ru/index.php',
        headers={
            'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10.15; rv:96.0) Gecko/20100101 Firefox/96.0',
            'Accept': 'application/json, text/javascript, */*; q=0.01',
            'Accept-Language': 'ru-RU,ru;q=0.8,en-US;q=0.5,en;q=0.3',
            'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
            'X-Requested-With': 'XMLHttpRequest',
            'Origin': 'https://racktables.yandex-team.ru',
            'Referer': 'https://racktables.yandex-team.ru/index.php?page=services&tab=projects',
            'Sec-Fetch-Dest': 'empty',
            'Sec-Fetch-Mode': 'cors',
            'Sec-Fetch-Site': 'same-origin',
        },
        cookies={
            'Session_id': session_id,
            'sessionid2': session_id,
            'PHPSESSID': php_sessid,
            'yandexuid': '3477267261621862552',
            'Cookie_check': 'CheckCookieCheckCookie',
        },
        data={
            'module': 'ajax',
            'ac': 'get-hbf-grid',
            'page': '1',
            'rows': '100',
            'sort': 'macro',
            'order': 'asc',
            'filterRules': json.dumps([
                {
                    'field': 'id',
                    'op': 'contains',
                    'value': project_id,
                },
            ]),
        },
    )
    result = rsp.json()
    P_ID_CACHE[project_id] = result
    return result


def get_firewall_info(oauth_token, macro):
    if macro in MACRO_CACHE:
        if MACRO_CACHE[macro] == 'raise-exception':
            raise UnknownProjectID(f'Project ID not found for {macro}')
        return MACRO_CACHE[macro]

    rsp = requests.get(
        f'https://racktables.yandex.net/export/project-id-networks.php?op=show_macro&name={macro}',
        headers={
            'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10.15; rv:96.0) Gecko/20100101 Firefox/96.0',
            'Accept': 'application/json, text/javascript, */*; q=0.01',
            'Accept-Language': 'ru-RU,ru;q=0.8,en-US;q=0.5,en;q=0.3',
            'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
            'X-Requested-With': 'XMLHttpRequest',
            'Origin': 'https://racktables.yandex-team.ru',
            'Referer': 'https://racktables.yandex-team.ru/index.php?page=services&tab=projects',
            'Sec-Fetch-Dest': 'empty',
            'Sec-Fetch-Mode': 'cors',
            'Sec-Fetch-Site': 'same-origin',
            'Authorization': f'OAuth {oauth_token}',
        },
    )
    try:
        info = rsp.json()
    except Exception:
        MACRO_CACHE[macro] = 'raise-exception'
        raise UnknownProjectID(f'Project ID not found for {macro}')
    MACRO_CACHE[macro] = info
    return info


def try_do(func, always_raise=False):
    tries = 5
    while tries > 0:
        tries = tries - 1
        try:
            return func()
        except Exception:
            if always_raise:
                raise


def get_abc_info(oauth_token, abc_id):
    if abc_id in ABC_INFO:
        return ABC_INFO[abc_id]

    def foo():
        next = (
            'https://abc-back.yandex-team.ru/api/v3/services/members/?role__scope=administration&role__scope=services_management&service=%s'
            % abc_id
        )
        res = {
            'abc_admins': [],
            'abc_managers': [],
        }

        while next is not None:
            r = requests.get(next, headers={
                'Authorization': f'OAuth {oauth_token}',
            })
            if r.status_code != 200 and r.status_code < 500:
                # print('abc service %s was not found (%s)' % (abc_id, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            next = j['next']

            res['abc_admins'] = res['abc_admins'] + [
                a['person']['login'] for a in j['results'] if a['role']['scope']['slug'] == 'administration'
            ]
            res['abc_managers'] = res['abc_managers'] + [
                a['person']['login'] for a in j['results'] if a['role']['scope']['slug'] == 'services_management'
            ]

        return res

    res = try_do(foo)
    ABC_INFO[abc_id] = res
    return res
