from collections import defaultdict
import json
import re
import sys
import typing


RE_CONSUMERS = re.compile(r'^[\w.-]+(?:,[\w.-]+)*$')
RE_REQUIRED_GRANTS = re.compile(r'^[\w.*-]+(?:,[\w.-]+)*$')
RE_TVM_CLIENT_ID = re.compile(r'^(?:\d{1,15}|-)$')  # 15 с потолка
RE_IP_PRIMITIVE = re.compile('^(?:[0-9.]+|[0-9a-fA-F:]+)$')


def warn(msg: str):
    print(msg, file=sys.stderr)


class ParseError(Exception):
    pass


class CompositeRecordKey(typing.NamedTuple):
    consumers: str
    required_grants: str
    tvm_client_id: str


class Record:
    def __init__(self):
        self.ips: typing.Set[str] = set()
        self.count = 0


TRecords = typing.Dict[CompositeRecordKey, Record]


def process_line(records: TRecords, binary_line: bytes):
    try:
        line = binary_line.decode('utf-8')
    except UnicodeDecodeError as err:
        raise ParseError('UnicodeDecodeError {}'.format(err))

    try:
        tskv = line.split(' tskv\t', 1)[1]
    except IndexError:
        raise ParseError('String is not "tskv"-tokenized')

    tskv_data = {}
    for kv in tskv.split('\t'):
        try:
            k, v = kv.split('=', 1)
        except ValueError:
            raise ParseError('Wrong tskv format in "{}"'.format(kv))
        if k in tskv_data:
            raise ParseError('Duplicate field "{}" in TSKV'.format(k))
        tskv_data[k] = v

    try:
        key = CompositeRecordKey(
            consumers=tskv_data['consumers'],
            required_grants=tskv_data['required_grants'],
            tvm_client_id=tskv_data['tvm_client_id'],
        )
        ip = tskv_data['ip']
    except KeyError as err:
        raise ParseError('Inconsistent record: KeyError {}'.format(err))

    if not RE_CONSUMERS.match(key.consumers):
        raise ParseError('Invalid consumers: "{}"'.format(key.consumers))
    if not RE_REQUIRED_GRANTS.match(key.required_grants):
        raise ParseError('Invalid grants: "{}"'.format(key.required_grants))
    if not RE_TVM_CLIENT_ID.match(key.tvm_client_id):
        raise ParseError('Invalid TVM id: "{}"'.format(key.tvm_client_id))
    if not RE_IP_PRIMITIVE.match(ip):
        raise ParseError('Invalid IP: "{}"'.format(ip))

    rec = records[key]
    rec.count += 1
    rec.ips.add(ip)


def parse_stdin() -> TRecords:
    records: TRecords = defaultdict(Record)
    for i, line in enumerate(sys.stdin.buffer, 1):
        try:
            process_line(records, line.rstrip(b'\n'))
        except ParseError as err:
            warn('Error in stdin line {} "{}": {}'.format(i, line, err))
    return records


def make_output(records: TRecords):
    for key, rec in records.items():
        print(json.dumps({
            'consumers': key.consumers,
            'required_grants': key.required_grants,
            'tvm_client_id': key.tvm_client_id,
            'ips': list(rec.ips),
            'count': rec.count,
        }))


def main():
    records = parse_stdin()
    make_output(records)


if __name__ == '__main__':
    main()
