from math import (
    ceil,
    floor,
)


# если для метрики задан агрегат из этого множества, то точки отсортируются
REQUIRES_SORTED_DATA = {'median', 'p_50', 'p_90', 'p_95', 'p_98', 'p_99'}

# если точки будут отсортированы, то эти агрегаторы будут использовать оптимизированные алгоритмы
BENEFITS_FROM_SORTED_DATA = {'min', 'max', 'non_negative_derivative'}


def avg(xs):
    if not xs:
        return 0
    else:
        return sum(xs) / float(len(xs))


def percentile(xs, percent):
    k = (len(xs) - 1) * percent
    f = int(floor(k))
    c = int(ceil(k))
    if f == c:
        return xs[int(k)]
    d0 = xs[int(f)] * (c - k)
    d1 = xs[int(c)] * (k - f)
    return d0 + d1


def non_negative_derivative(xs, sorted=False):
    if sorted:
        _min = xs[0]
        _max = xs[-1]
    else:
        _min, _max = None, None
        for x in xs:
            if _min is None or x < _min:
                _min = x
            if _max is None or x > _max:
                _max = x

    derivative = _max - _min
    return max(0, derivative)


def find_min(xs, sorted=False):
    if sorted:
        return xs[0]
    else:
        return min(xs)


def find_max(xs, sorted=False):
    if sorted:
        return xs[-1]
    else:
        return max(xs)


def should_sort_data(requested_aggregates):
    return bool(set(requested_aggregates) & REQUIRES_SORTED_DATA)


def get_aggregation_f_by_name(name):
    mapping = {
        'sum': sum,
        'min': find_min,
        'max': find_max,
        'avg': avg,
        'median': lambda xs: percentile(xs, 0.5),
        'p_50': lambda xs: percentile(xs, 0.5),
        'p_90': lambda xs: percentile(xs, 0.9),
        'p_95': lambda xs: percentile(xs, 0.95),
        'p_98': lambda xs: percentile(xs, 0.98),
        'p_99': lambda xs: percentile(xs, 0.99),
        'non_negative_derivative': non_negative_derivative,
    }
    return mapping[name]
