import configparser

from passport.backend.tools.metrics.aggregates import get_aggregation_f_by_name
from passport.backend.tools.metrics.file_parsers import (
    re_log_parser,
    table_log_parser,
    tskv_log_parser,
)
from passport.backend.tools.metrics.metrics import MetricParser
from passport.backend.utils.dc import get_current_dc
import yenv


DEFAULT_TTL = 90


def strip_prefix(option):
    return option.split('.', 1)[1]


def get_values(option_value):
    return [v.strip() for v in option_value.split(',')]


def get_ini_parser():
    return configparser.ConfigParser()


def load_config(conf_path):
    config = get_ini_parser()
    config.read([conf_path])
    return config


def get_golovan_settings(config):
    default_settings = {
        'geo': get_current_dc(),
        'ttl': DEFAULT_TTL,
        'ctype': yenv.type,
    }
    options = config.options('golovan')
    settings = dict(default_settings, **{o: config.get('golovan', o) for o in options})
    return settings


def parse_config(config):
    parser_configs = []

    for section in config.sections():
        if section == 'golovan':
            continue

        filters = {}
        fields = {}
        zero_by_default = {}
        aggregate_fs = []
        metric_name_template = None
        metric_value_column = None
        scale_to_seconds_interval = None
        columns = None
        columns_delimiter = ' '
        columns_rewrite = {}
        ignore_empty_columns = True
        re_log_format = None

        for option in config.options(section):
            value = config.get(section, option).strip()

            if option.startswith('filters.neg.'):
                try:
                    filters[strip_prefix(strip_prefix(option))][1] = get_values(value)
                except KeyError:
                    filters[strip_prefix(strip_prefix(option))] = [[], get_values(value)]
            elif option.startswith('filters.'):
                try:
                    filters[strip_prefix(option)][0] = get_values(value)
                except KeyError:
                    filters[strip_prefix(option)] = [get_values(value), []]
            elif option.startswith('zero.'):
                zero_by_default[strip_prefix(option)] = get_values(value)
            elif option == 'fields':
                fields = get_values(value)
            elif option == 'metric':
                metric_name_template = value
            elif option == 'aggregate':
                aggregate_fs = get_values(value)
            elif option == 'metric.value':
                metric_value_column = value
            elif option == 'scale_to_seconds_interval':
                scale_to_seconds_interval = value
            elif option == 'columns':
                columns = get_values(value)
            elif option == 'columns.re':
                re_log_format = value
            elif option == 'columns.delimiter':
                columns_delimiter = value
                if len(columns_delimiter) > 1 and columns_delimiter.startswith('"') and columns_delimiter.endswith('"'):
                    columns_delimiter = columns_delimiter.strip('"')
            elif option == 'columns.ignore_empty':
                ignore_empty_columns = value.lower() in ['true', 't', 'yes', '1']
            elif option.startswith('rewrite.'):
                if option.endswith('.re'):
                    column = strip_prefix(option)[:-len('.re')]
                    try:
                        columns_rewrite[column]['pattern'] = value
                    except KeyError:
                        columns_rewrite[column] = {'pattern': value}
                elif option.endswith('.sub'):
                    column = strip_prefix(option)[:-len('.sub')]
                    try:
                        columns_rewrite[column]['sub'] = value
                    except KeyError:
                        columns_rewrite[column] = {'sub': value}

        if re_log_format:
            log_type = re_log_parser(re_log_format)
        elif columns:
            log_type = table_log_parser(columns, columns_delimiter, ignore_empty_columns)
        else:
            log_type = tskv_log_parser()

        parser_configs.append(dict(
            log_type=log_type,
            metric_name_template=metric_name_template,
            fields=fields,
            filters=filters,
            defaults=zero_by_default,
            aggregate_fs=aggregate_fs or ['sum'],
            metric_value_column=metric_value_column,
            scale_to_seconds_interval=scale_to_seconds_interval,
            columns_rewrite=columns_rewrite,
        ))

    return parser_configs


def get_metric_parsers(config, debug_qb2=False, debug_aggregate=False):
    parser_configs = parse_config(config)
    parsers = []

    for parser_config in parser_configs:
        parsers.append(
            MetricParser(
                log_type=parser_config['log_type'],
                metric_name_template=parser_config['metric_name_template'],
                fields=parser_config['fields'],
                filters=parser_config['filters'],
                defaults=parser_config['defaults'],
                aggregate_fs=[(n, get_aggregation_f_by_name(n)) for n in parser_config['aggregate_fs']],
                metric_value_column=parser_config['metric_value_column'],
                scale_to_seconds_interval=parser_config['scale_to_seconds_interval'],
                debug_qb2=debug_qb2,
                debug_aggregate=debug_aggregate,
                columns_rewrite=parser_config['columns_rewrite'],
            ),
        )

    return parsers
