import csv
import re

from passport.backend.utils.common import merge_dicts


def parse_tskv_line(line):
    columns = line.split('\t')
    record = {}
    for kv in columns:
        try:
            key, value = kv.split('=', 1)
        except ValueError:
            continue
        record[key] = value.rstrip()
    return record


def tskv_log_parser():

    def parser(stream):
        for line in stream:
            yield parse_tskv_line(line)

    return parser


def parse_csv_line(line, delimiter, ignore_empty_columns):
    parser = csv.reader([line], delimiter=delimiter)
    row = next(parser, None)
    if ignore_empty_columns and row is not None:
        return filter(None, row)
    return row


def parse_re_line(line, regular_expression):
    match = regular_expression.match(line.rstrip())
    if match:
        return match.groupdict()
    return {}


def re_log_parser(regular_expression):
    compiled_re = re.compile(regular_expression)
    base_dict = {k: None for k in compiled_re.groupindex}

    def parser(stream):
        for line in stream:
            yield merge_dicts(base_dict, parse_re_line(line, compiled_re))

    return parser


def table_log_parser(columns, delimiter, ignore_empty_columns):
    base_dict = {k: None for k in columns}

    def parser(stream):
        for line in stream:
            pairs = zip(
                columns,
                parse_csv_line(line, delimiter, ignore_empty_columns),
            )
            yield merge_dicts(base_dict, pairs)
    return parser
