# cython: language_level=3
import argparse
import io
import sys
import threading

from passport.backend.tools.metrics import flock_utils
from passport.backend.tools.metrics.conf import (
    get_golovan_settings,
    get_metric_parsers,
    load_config,
)
from passport.backend.tools.metrics.golovan_pull import GolovanPull
from passport.backend.tools.metrics.golovan_push import (
    format_metrics_for_golovan_push,
    GolovanPush,
)
from passport.backend.tools.metrics.io_utils import (
    print_stderr,
    print_stdout,
)
from passport.backend.tools.metrics.runner import gather_metrics
from passport.backend.tools.metrics.tail import tail
from passport.backend.tools.metrics.timer import PerpetualTimer
from passport.backend.utils import flock


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-c', '--conf', required=True)
    parser.add_argument('-f', '--file', required=False)
    parser.add_argument('--show-golovan-tags', required=False, action='store_true')
    parser.add_argument('--debug-lock', required=False, action='store_true')
    parser.add_argument('--debug-qb2', required=False, action='store_true')
    parser.add_argument('--debug-aggregate', required=False, action='store_true')
    parser.add_argument('--debug-push', required=False, action='store_true')
    parser.add_argument('--debug-pull', required=False, action='store_true')
    parser.add_argument('--lock', required=False, nargs='?', const='')
    parser.add_argument('--push-to', required=False)
    parser.add_argument('--listen', required=False)
    return parser.parse_args()


def print_golovan_settings(golovan_settings):
    for k, v in sorted(golovan_settings.items(), key=lambda pair: pair[0]):
        print_stdout(k, '=', str(v))


def do_meaningful_work(args):
    config = load_config(args.conf)

    golovan_settings = get_golovan_settings(config)

    if args.show_golovan_tags:
        print_golovan_settings(golovan_settings)
        sys.exit(0)

    golovan_ttl = golovan_settings.pop('ttl')
    metrics = get_metric_parsers(config, args.debug_qb2, args.debug_aggregate)

    if args.file:
        stream = tail(args.file)
    else:
        stream = io.TextIOWrapper(sys.stdin.buffer, encoding='utf-8', errors='ignore')

    if args.push_to:
        golovan = GolovanPush(args.push_to, args.debug_push)
        gathering_thread = threading.Thread(
            name='gathering-thread',
            target=lambda: list(gather_metrics(stream, metrics, buffer_size=1)),
        )
        agg_thread = PerpetualTimer(
            interval=5.0,
            function=golovan.aggregate_and_push_metrics,
            args=(metrics, golovan_settings),
            kwargs={},
            name='pushing-thread',
        )
        gathering_thread.daemon = True
        gathering_thread.start()
        agg_thread.start()
        gathering_thread.join()
    elif args.listen:
        host, _, port = args.listen.partition(':')
        port = int(port)
        golovan = GolovanPull(host, port, metrics, golovan_settings, args.debug_pull)
        gathering_thread = threading.Thread(
            name='gathering-thread',
            target=lambda: list(gather_metrics(stream, metrics, buffer_size=1)),
        )
        if args.debug_pull:
            print_stderr('Starting gathering thread...')
        gathering_thread.daemon = True
        gathering_thread.start()
        if args.debug_pull:
            print_stderr('Starting serving thread...')
        golovan.start_server()
    else:
        kv_metric_value = {k: v for k, v in gather_metrics(stream, metrics)}
        golovan_formatted_metrics = format_metrics_for_golovan_push(
            metrics=kv_metric_value,
            ttl=golovan_ttl,
            **golovan_settings
        )
        print_stdout(golovan_formatted_metrics)


def maybe_run_under_lock(args, f):
    if args.lock is not None:
        lock_filename = args.lock
        if not lock_filename:
            lock_filename = flock_utils.get_lock_name('/tmp/passport-metrics/', args.conf)
        try:
            if args.debug_lock:
                print('Acquiring lock {}'.format(repr(lock_filename)), file=sys.stderr)
            with flock.acquire(lock_filename):
                f()
        except flock.AlreadyLocked:
            if args.debug_lock:
                print('Cannot acquire lock. File {} is already locked'.format(repr(lock_filename)), file=sys.stderr)
            sys.exit(1)
    else:
        if args.debug_lock:
            print('Running without locks', file=sys.stderr)
        f()


def run_app():
    try:
        args = parse_args()
        maybe_run_under_lock(args, lambda: do_meaningful_work(args))
    except KeyboardInterrupt:
        sys.exit(1)


if __name__ == '__main__':
    run_app()
