# cython: language_level=3
import copy
from itertools import product
from pprint import pformat
import re
import sys
import threading

from passport.backend.tools.metrics.aggregates import (
    BENEFITS_FROM_SORTED_DATA,
    should_sort_data,
)
from passport.backend.utils.common import merge_dicts


RE_STR_FORMAT = re.compile(r'{.+?}')


RESERVED_NAMES = {'__aggregate__'}


def get_template_vars(template):
    # template - строка с синтаксисом str.format
    # возвращает словарь ключ - его оригинальная запись
    # например '{foo} bar' вернёт {'foo': 'foo'}
    template_vars = RE_STR_FORMAT.findall(template)
    return {v.split(':')[0].lstrip('{').rstrip('}'): v for v in template_vars}


def if_matches(filters, content):
    for key, (possible_values, impossible_values) in filters.items():
        try:
            if possible_values and content[key] not in possible_values:
                return False
            elif not possible_values and key not in content:
                return False
            if impossible_values and content[key] in impossible_values:
                return False
        except KeyError:
            return False
    return True


class MetricParser(object):
    def __init__(self,
                 log_type,
                 metric_name_template,
                 fields,
                 filters,
                 defaults,
                 aggregate_fs,
                 metric_value_column,
                 extra_context=None,
                 scale_to_seconds_interval=None,
                 columns_rewrite=None,
                 debug_qb2=False,
                 debug_aggregate=False,
                 ):
        self.log_parser = log_type
        self.metric_name_template = metric_name_template
        self.fields = set(fields)
        self.filters = filters
        self.defaults = defaults
        self.aggregate_fs = aggregate_fs
        self.metric_value_column = metric_value_column
        self.scale_to_seconds_interval = scale_to_seconds_interval
        self.columns_rewrite = None
        self.debug_qb2 = debug_qb2
        self.debug_aggregate = debug_aggregate

        self.values_lock = threading.Lock()

        # экстра-переменные, доступные для использования в именах метрики
        self.extra_context = {}
        # словарь метрика: массив точек
        self.values = {}

        self.init_columns_rewrite(columns_rewrite or {})
        self.init_extra_context(extra_context)
        self.init_filters()

    def init_columns_rewrite(self, columns_rewrite):
        self.columns_rewrite = {}
        for column, re_params in columns_rewrite.items():
            pattern = re.compile(re_params['pattern'])
            repl = re_params['sub']
            self.columns_rewrite[column] = lambda value: re.sub(pattern, repl, value)

    def init_extra_context(self, extra_context):
        extra_context = extra_context or {}

        # сюда складываются все шаблонные переменные из имени метрики,
        # чтобы можно было сделать частичный str.format
        # на самом деле он будет не частичный, например:
        # '{x} {y}'.format(x=10, y='{y}')
        # юзкейс: например переменная __aggregate__ в имени метрики
        template_vars = get_template_vars(self.metric_name_template).items()
        for var_key, var_key_and_opts in template_vars:
            self.extra_context[var_key] = var_key_and_opts

        self.extra_context.update(extra_context)

    def init_filters(self):
        expected_fields = (self.extra_context.keys() | self.fields) - RESERVED_NAMES
        for field in expected_fields:
            if field not in self.filters:
                self.filters[field] = [[], []]

    def get_value(self, record):
        if self.metric_value_column:
            # если метрика - это значение столбца
            return float(record[self.metric_value_column])
        else:
            # количество строк в противном случае
            return 1

    def fill_empty_metrics(self, values):
        # если метрика без переменных в шаблоне, зануляем её
        if not set(self.extra_context) - RESERVED_NAMES:
            if self.metric_name_template not in values:
                values[self.metric_name_template] = [0]

        # процедура заполнения конкретных метрик нулями для обеспечения непрерывности метрики
        if not self.defaults:
            return

        # делается декартово произведение всех ключей и значений из конфига
        product_args = []
        for field, expected_values in self.defaults.items():
            product_args.append([(field, ev) for ev in expected_values])

        for kwargs in product(*product_args):
            kwargs = dict(kwargs)
            key = self.metric_name_template.format(**merge_dicts(self.extra_context, kwargs))
            if key not in values:
                values[key] = [0]

    def accumulate(self, input_stream):
        # процедура накопления значений по метрикам
        # она запускается несколько раз за один лог, поэтому не должна сбрасывать состояние
        for record in self.log_parser(input_stream):
            if self.debug_qb2:
                print(pformat(record), file=sys.stderr)
            if not if_matches(self.filters, record):
                continue
            # перепись значений столбцов, если требуется
            if self.columns_rewrite:
                for column, rewrite_f in self.columns_rewrite.items():
                    record[column] = rewrite_f(record[column])

            try:
                metric_value = self.get_value(record)
            except ValueError:
                continue
            safe_record = {k: v.replace('{', '{{').replace('}', '}}') for k, v in record.items()}
            metric = self.metric_name_template.format(**merge_dicts(self.extra_context, safe_record))
            with self.values_lock:
                try:
                    self.values[metric].append(metric_value)
                except (AttributeError, KeyError):
                    self.values[metric] = [metric_value]

    def get_metrics(self):
        with self.values_lock:
            new_values = copy.copy(self.values)
            self.values = {}
        self.fill_empty_metrics(new_values)

        # надо сортировать массив значений, если какая либо агрегирующая функция требует отсортированных данных
        aggregate_f_names = [aggregate_f_name for aggregate_f_name, _aggregate_f in self.aggregate_fs]
        sort_data = should_sort_data(aggregate_f_names)
        if self.debug_aggregate:
            print('Aggregation functions:', pformat(aggregate_f_names), file=sys.stderr)
            print('Will sort data:', sort_data, file=sys.stderr)

        if self.scale_to_seconds_interval:
            interval_length = float(self.scale_to_seconds_interval)
        else:
            interval_length = 0

        for metric_name_template, values in new_values.items():
            if sort_data:
                values.sort()

            for aggregate_f_name, aggregate_f in self.aggregate_fs:
                metric_name = metric_name_template.format(
                    **merge_dicts(
                        self.extra_context,
                        {
                            '__aggregate__': aggregate_f_name,
                        },
                    )
                )
                if self.debug_aggregate:
                    print('Function:', aggregate_f_name, file=sys.stderr)
                    print('Values:', pformat(values), file=sys.stderr)
                if aggregate_f_name in BENEFITS_FROM_SORTED_DATA:
                    # если агрегат считать лучше на отсортированных данных, то инструктиурем функцию о том,
                    # что данные уже отсортированы
                    aggregated_value = aggregate_f(values, sorted=sort_data)
                else:
                    # в противном случае считаем наивным алгоритмом
                    aggregated_value = aggregate_f(values)
                if self.scale_to_seconds_interval:
                    # приводим метрику к rps, если надо
                    aggregated_value /= interval_length
                yield metric_name, aggregated_value
