import json
import logging
from urllib.parse import urljoin

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import BaseCoreError


log = logging.getLogger('passport.monconf')


class BaseJugglerError(BaseCoreError):
    pass


class JugglerTemporaryError(BaseJugglerError):
    pass


class JugglerPermanentError(BaseJugglerError):
    pass


class Juggler(BaseBuilder, JsonParserMixin):
    base_error_class = BaseJugglerError
    temporary_error_class = JugglerTemporaryError
    parser_error_class = JugglerPermanentError

    def __init__(self):
        super().__init__(
            url=settings.JUGGLER_URL,
            timeout=3,
            retries=1,
            logger=log,
        )

    def _error_detector(self, data, raw_response):
        if not data.get('success'):
            raise JugglerPermanentError('Juggler error: {}'.format(data))

    def add_or_update_check(self, token, data):
        return self._request_with_retries_simple(
            method='POST',
            params=dict(
                do='1',
            ),
            headers={
                'Authorization': 'OAuth {}'.format(token),
                'Content-type': 'application/json',
            },
            data=json.dumps(data),
            url=urljoin(self.url, 'api/checks/add_or_update'),
            error_detector=None,
            parser=self.parse_json,
        )


def get_juggler():
    return Juggler()  # pragma: no cover
