import json
import logging
from urllib.parse import urljoin

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import BaseCoreError


log = logging.getLogger('passport.monconf')


class BaseSolomonError(BaseCoreError):
    pass


class SolomonTemporaryError(BaseSolomonError):
    pass


class SolomonPermanentError(BaseSolomonError):
    pass


class Solomon(BaseBuilder, JsonParserMixin):
    base_error_class = BaseSolomonError
    temporary_error_class = SolomonTemporaryError
    parser_error_class = SolomonPermanentError

    def __init__(self, token=None):
        super().__init__(
            url=urljoin(settings.SOLOMON_URL, 'api/v2/projects/{}/'.format(settings.SOLOMON_PROJECT)),
            timeout=3,
            retries=1,
            logger=log,
            tvm_dst_alias=None if token else settings.SOLOMON_TVM_ALIAS,
        )
        self.token = token

    def _simple_error_detector(self, data, raw_response):
        if raw_response.status_code != 200:
            raise SolomonPermanentError('Solomon error: {}'.format(data))

    def _make_request(self, url, method, error_detector, params=None, headers=None, data=None):
        params = params or {}
        headers = headers or {}
        headers['Content-type'] = 'application/json'
        if self.token:
            headers['Authorization'] = 'OAuth {}'.format(self.token)
        return self._request_with_retries_simple(
            error_detector=error_detector,
            parser=self.parse_json,
            method=method,
            url_suffix=url,
            params=params,
            data=data,
            headers=headers,
        )

    def get_alert_by_id(self, id_):
        return self._make_request(
            url='alerts/{}'.format(id_),
            method='GET',
            error_detector=self._simple_error_detector,
        )

    def add_or_update_alert(self, data):
        id_ = data['id']
        try:
            alert = self.get_alert_by_id(id_)
        except SolomonPermanentError:
            alert = None

        if alert:
            data['version'] = alert['version']
            return self._make_request(
                url='alerts/{}'.format(id_),
                method='PUT',
                data=json.dumps(data),
                error_detector=self._simple_error_detector,
            )
        else:
            return self._make_request(
                url='alerts',
                method='POST',
                data=json.dumps(data),
                error_detector=self._simple_error_detector,
            )


def get_solomon(token=None):
    return Solomon(token=token)  # pragma: no cover
