from typing import Optional

import dataclasses
from passport.backend.tools.monconf.settings import (
    JUGGLER_PROJECT,
    SOLOMON_PROJECT,
)
from passport.backend.utils.common import noneless_dict


@dataclasses.dataclass
class JugglerNotification:
    template_name: str
    template_kwargs: dict

    def to_dict(self):
        return dict(
            template_name=self.template_name,
            template_kwargs=self.template_kwargs,
        )


@dataclasses.dataclass
class JugglerCheck:
    host: str
    service: str
    description: str
    tags: list[str]
    notifications: list[JugglerNotification]
    project: str = JUGGLER_PROJECT
    refresh_time: int = 5
    ttl: int = 900
    aggregator: str = "logic_or"
    nodata_mode: str = "force_crit"
    flaps_stable_time: int = 180
    flaps_critical_time: int = 180
    flaps_boost_time: int = 180
    meta: Optional[dict] = None

    def to_dict(self):
        return dict(
            project=self.project,
            host=self.host,
            service=self.service,
            refresh_time=self.refresh_time,
            ttl=self.ttl,
            description=self.description,
            aggregator=self.aggregator,
            aggregator_kwargs=dict(
                nodata_mode=self.nodata_mode,
            ),
            flaps=dict(
                stable_time=self.flaps_stable_time,
                critical_time=self.flaps_critical_time,
                boost_time=self.flaps_boost_time,
            ),
            meta=self.meta,
            tags=self.tags,
            notifications=[n.to_dict() for n in self.notifications],
        )


@dataclasses.dataclass
class SolomonChannel:
    id: str
    statuses: list[str]
    repeat_delay_sec: int = 5 * 60

    def to_dict(self):
        return dict(
            id=self.id,
            config=dict(
                notifyAboutStatuses=self.statuses,
                repeatDelaySecs=self.repeat_delay_sec,
            )
        )


@dataclasses.dataclass
class SolomonThreshold:
    predicate: str  # EQ/GT/GTE/LT/LTE/NE/UNRECOGNIZED
    selectors: str
    threshold: float
    time_aggregation: str  # AT_ALL_TIMES/AT_LEAST_ONE/AVG/COUNT/LAST_NON_NAN/MAX/MIN/SUM/UNRECOGNIZED

    def to_dict(self):
        return dict(
            predicate=self.predicate,
            selectors=self.selectors,
            threshold=self.threshold,
            timeAggregation=self.time_aggregation,
        )


@dataclasses.dataclass
class SolomonAlert:
    id: str
    name: str
    channels: list[SolomonChannel]
    threshold: SolomonThreshold
    project_id: str = SOLOMON_PROJECT
    period_millis: int = 5 * 60 * 1000
    state: str = 'ACTIVE'

    def to_dict(self):
        return noneless_dict(
            id=self.id,
            name=self.name,
            channels=[c.to_dict() for c in self.channels],
            type=dict(threshold=self.threshold.to_dict()),
            projectId=self.project_id,
            periodMillis=self.period_millis,
            state=self.state,
        )
