import logging

from passport.backend.core.conf import settings
from passport.backend.tools.monconf.builders.juggler import get_juggler
from passport.backend.tools.monconf.builders.solomon import get_solomon
from passport.backend.tools.monconf.classes import (
    JugglerCheck,
    SolomonAlert,
    SolomonChannel,
    SolomonThreshold,
)
from passport.backend.tools.monconf.errors import ConfigError


log = logging.getLogger('passport.monconf')


class LogbrokerProcessor:

    def __init__(self, juggler_token, solomon_token):
        self.juggler_token = juggler_token
        self.solomon_token = solomon_token

    def process_logbroker_read_lag(self, config: dict, target_conf: dict):
        topic = config['topic']
        consumer = config['consumer']
        service = target_conf['name']
        description = target_conf['description']
        tags = ['urgency-level-{}'.format(target_conf['urgency'])]
        tags.extend(target_conf.get('extra_tags', []))
        cluster = target_conf['cluster']
        juggler_check_data = JugglerCheck(
            host=settings.JUGGLER_HOST,
            service=service,
            description=description,
            tags=tags,
            notifications=[],
            meta={'creator': 'passport-monconf'},
            flaps_stable_time=target_conf['antiflap'],
            flaps_boost_time=target_conf['antiflap'],
            flaps_critical_time=target_conf['antiflap'],
        ).to_dict()
        log.debug('Creating Juggler check {}'.format(juggler_check_data))
        juggler_res = get_juggler().add_or_update_check(
            token=self.juggler_token,
            data=juggler_check_data,
        )
        log.debug('Juggler reply: {}'.format(juggler_res))

        solomon_alert_data = SolomonAlert(
            id=service,
            name=description,
            channels=[
                SolomonChannel(
                    id=settings.SOLOMON_TO_JUGGLER_CHANNEL,
                    statuses=['ALARM', 'WARN', 'OK', 'NO_DATA', 'ERROR'],
                ),
            ],
            threshold=SolomonThreshold(
                predicate='GTE',
                selectors="""
                {{
                    project='kikimr',
                    cluster='{cluster}',
                    service='pqtabletAggregatedCounters',
                    Account='passport',
                    OriginDC='Iva|Man|Myt|Sas|Vla',
                    host='Iva|Man|Myt|Sas|Vla',
                    sensor='ReadTimeLagMs',
                    ConsumerPath='{consumer}',
                    TopicPath='{topic}',
                    partition='-',
                    user_counters='PersQueue'
                }}
                """.format(cluster=cluster, consumer=consumer, topic=topic),
                threshold=target_conf['threshold'],
                time_aggregation='AT_ALL_TIMES',
            ),
        ).to_dict()
        log.debug('Creating Solomon alert {}'.format(solomon_alert_data))
        solomon_res = get_solomon(token=self.solomon_token).add_or_update_alert(
            data=solomon_alert_data,
        )
        log.debug('Solomon reply: {}'.format(solomon_res))

    def process(self, config: dict):
        for target, target_conf in config['targets'].items():
            if target == 'read-lag':
                self.process_logbroker_read_lag(config, target_conf)
            else:
                raise ConfigError('Wrong target type {}'.format(target))
