import argparse
import logging
import os
from typing import Optional

from passport.backend.core.conf import settings
from passport.backend.tools.monconf import (
    secrets,
    settings as monconf_settings,
)
from passport.backend.tools.monconf.errors import ConfigError
from passport.backend.tools.monconf.logbroker import LogbrokerProcessor
import yaml


log = logging.getLogger('passport.monconf')
MAX_CONFIGS_LOOKUP_DEPTH = 10


def process_config(config_file: str, solomon_token: Optional[str], juggler_token: str):
    log.info('Processing config {}'.format(config_file))
    with open(config_file) as f:
        config = yaml.load(f, Loader=yaml.SafeLoader)

    config_type = config.get('type')
    log.info('Config type is {}'.format(config_type))
    if config_type == 'logbroker':
        LogbrokerProcessor(
            solomon_token=solomon_token,
            juggler_token=juggler_token,
        ).process(config)
    else:
        raise ConfigError('Wrong config type {}'.format(config_type))


def list_configs(config_dir: str, depth=0):
    if depth > MAX_CONFIGS_LOOKUP_DEPTH:
        raise RuntimeError('Max config dir depth reached at {}'.format(config_dir))
    for sub_path in os.listdir(config_dir):
        path = os.path.join(config_dir, sub_path)
        if any(path.endswith(ext) for ext in ('.conf', '.yml', '.yaml')):
            yield path
        if os.path.isdir(path):
            yield from list_configs(path, depth + 1)


def run_app():
    parser = argparse.ArgumentParser(description='Configure monitoring')
    parser.add_argument('-V', '--less-verbosity', action='store_true')
    parser.add_argument('--config-dir', default='/etc/yandex/passport-monconf')
    parser.add_argument('--juggler-token-file', type=str)
    parser.add_argument('--solomon-token-file', type=str)

    settings.configure(monconf_settings)

    args = parser.parse_args()

    juggler_token = secrets.JUGGLER_TOKEN
    if args.juggler_token_file:
        with open(args.juggler_token_file) as f:
            juggler_token = f.read().rstrip()

    if not juggler_token:
        raise RuntimeError('No Juggler token is provided via secrets or via CLI')

    solomon_token = None
    if args.solomon_token_file:
        with open(args.solomon_token_file) as f:
            solomon_token = f.read().rstrip()

    if not solomon_token:
        log.debug('No Solomon token is provided via CLI')

    log_level = 'INFO' if args.less_verbosity else 'DEBUG'
    log.setLevel(log_level)
    out_handler = logging.StreamHandler()
    out_handler.setLevel(log_level)
    log.addHandler(out_handler)

    processed = 0
    errors = 0
    try:
        for config_file in list_configs(args.config_dir):
            try:
                process_config(config_file, solomon_token=solomon_token, juggler_token=juggler_token)
            except ConfigError as err:
                errors += 1
                raise ConfigError('{} in file {}'.format(err, config_file))
            finally:
                processed += 1

    finally:
        log.info('Processed {} configs{}'.format(
            processed,
            ', {} errors'.format(errors) if errors else ''
        ))

    if errors:
        exit(255)
