# -*- coding: utf-8 -*-

from __future__ import print_function

import logging
import sys

from passport.backend.api.settings import settings as p_settings
from passport.backend.core.conf import (
    DynamicSettings,
    settings,
)
from passport.backend.core.builders.historydb_api import HistoryDBApi
from passport.backend.utils.time import get_unixtime


logger = logging.getLogger(__name__)


def main():
    logging.basicConfig(level=logging.INFO)

    if not settings.configured:
        dynamic_settings = DynamicSettings(
            py_filename='/usr/lib/yandex/passport-settings/overridden_settings.py',
            import_as_name='passport_overridden_settings',
        )
        settings.configure(settings=p_settings, dynamic_settings=dynamic_settings)

    ha = HistoryDBApi(timeout=30, retries=4)

    for line in sys.stdin:
        uid = int(line.strip())
        logger.info('uid=' + str(uid))

        try:
            events = ha.events(
                uid,
                1,
                get_unixtime(),
                name='info.is_maillist',
                ascending=False,
            )
            if not is_maillist(uid, events):
                print(uid)
        except Exception:
            skip_line = 'Skip uid: %s' % uid
            print(skip_line)
            logger.exception(skip_line)


def is_maillist(uid, events):
    # assert events отсортированы обратном хронологическом порядке
    retval = False
    for event in events:
        if event['name'] == 'info.is_maillist':
            value = event.get('value')
            if value == '1' or value is None:
                # Самая поздняя запись это включение атрибута, или удаление
                # атрибута (запись об удалении появляется, только если он был
                # до этого включён)
                retval = True
                break
            elif value == '0':
                # Самая поздняя запись это отключение атрибута
                retval = False
                break
            else:
                raise NotImplementedError('Unexpected info.is_maillist event: %s, %s' % (uid, str(event)))
    return retval


if __name__ == '__main__':
    main()
