#/usr/bin/env python3

"""
Accepts a file with three columns (uid, attr_id, value) as input and forms SQL commands
to remove attributes by chunks. Ignore extra columns in the file
"""

import argparse
import csv
import sys


class ArgsParser(argparse.ArgumentParser):
    def __init__(self, *args, **kwargs):
        super().__init__(
            formatter_class=argparse.RawDescriptionHelpFormatter,
            description=__doc__,
            *args,
            **kwargs,
        )

        self.add_argument('inp_file', metavar='input_filename', type=argparse.FileType(), help='Input filename or - for stdin')
        self.add_argument('out_file', metavar='output_filename', default='-', type=argparse.FileType('w'), help='Output filename or - for stdout')
        self.add_argument('-c', dest='chunk_size', type=int, default=1000, help='Chunk size (default: 1000)')
        self.add_argument('-d', dest='delimiter', default='\t', help='Column delimiter (default: \\t)')
        self.add_argument('-k', dest='skip_value', action='store_true', default=False, help='Use only uid and type_id')
        self.add_argument('-w', dest='where_clause', default='', help='An additional condition for where clause')

    def error(self, message, stderr=None):
        stderr = stderr or sys.stderr
        print('error: %s' % message, file=stderr)
        print('-' * 20, file=stderr)
        self.print_help(file=stderr)
        sys.exit(2)


def main():
    args_parser = ArgsParser()
    args = args_parser.parse_args()

    chunk_size = args.chunk_size

    csv_reader = csv.reader(args.inp_file, delimiter=args.delimiter)

    out_file = args.out_file
    def print_(line):
        print(line, file=out_file)

    where_clause = ''
    if args.where_clause:
        where_clause = ' AND ' + args.where_clause

    for i, row in enumerate(csv_reader):
        if i % chunk_size == 0:
            if i > 0:
                print_('COMMIT;')
            print_('BEGIN;')
        value_clause = ''
        if not args.skip_value:
            value_clause = ' AND value = \'{}\''.format(row[2].replace('\'', '\\\''))

        print_('DELETE FROM attributes WHERE uid = {uid:d} AND type = {type:d}{value_clause}{where_clause};'.format(
            uid=int(row[0]),
            type=int(row[1]),
            value_clause=value_clause,
            where_clause=where_clause,
        ))
    print_('COMMIT;')


if __name__ == '__main__':
    main()
