# -*- coding: utf-8 -*-
import itertools
import json

from passport.backend.tools.shardushka.exceptions import ShardushkaError
from passport.backend.tools.shardushka.range import Range
from passport.backend.tools.shardushka.rangeset import RangeSet
from passport.backend.tools.shardushka.shard import Shard


class Config:
    def __init__(self, config_path):
        self._config_path = config_path
        self._shards_by_id = {}
        self._rangesets_by_name = {}

    @property
    def ordered_rangesets(self):
        return sorted(
            self._rangesets_by_name.items(),
            key=lambda item: item[1].last_range().min_uid,
        )

    def load(self):
        """
        Читает конфиг из файла
        """
        with open(self._config_path) as f:
            data = json.load(f)
        for shard_params in data['shards']:
            self.create_shard(shard_id=shard_params['id'])
        for rangeset_name, ranges in data['rangesets'].items():
            self.create_rangeset(name=rangeset_name)
            for range_params in ranges:
                self.create_range(
                    shard_id=range_params['shard'],
                    rangeset_name=rangeset_name,
                    min_uid=range_params['min'],
                )

    def dump(self):
        """
        Сохраняет конфиг в файл
        """
        config = {
            'shards': [
                shard.as_dict()
                for shard in self._shards_by_id.values()
            ],
            'rangesets': {
                name: [
                    range_.as_dict()
                    for range_ in rangeset
                ]
                for name, rangeset in self.ordered_rangesets
            }
        }
        with open(self._config_path, 'w') as f:
            json.dump(config, f, indent=2)
            f.write('\n')

    def create_shard(self, shard_id):
        """
        Создаёт шард
        :param shard_id: номер шарда
        """
        if shard_id in self._shards_by_id:
            raise ShardushkaError(f'Shard {shard_id} already exists')
        self._shards_by_id[shard_id] = Shard(shard_id=shard_id)

    def create_rangeset(self, name):
        """
        Создаёт пустой rangeset (множество диапазонов)
        :param name: имя rangeset'а
        """
        if name in self._rangesets_by_name:
            raise ShardushkaError(f'Rangeset `{name}` already exists')
        self._rangesets_by_name[name] = RangeSet(name=name)

    def create_range(self, rangeset_name, shard_id, min_uid=None, capacity=None):
        """
        Создаёт один открытый сверху диапазон в указанном rangeset'е. Необходимо указать либо начало диапазона,
        либо вместимость (тогда началом будет начало последнего открытого диапазона)
        :param rangeset_name: имя rangeset'а, в котором создавать диапазон
        :param shard_id: id шарда, для которого создавать диапазон
        :param min_uid: uid, с которого должен начинаться диапазон
        :param capacity: размер (число уидов) добавляемого диапазона
        """
        if shard_id not in self._shards_by_id:
            raise ShardushkaError(f'Shard {shard_id} not found')

        rangeset = self._rangesets_by_name.get(rangeset_name)
        if rangeset is None:
            raise ShardushkaError(f'Rangeset `{rangeset_name}` not found')

        if min_uid is not None:
            range_ = Range(shard_id=shard_id, min_uid=min_uid)
        elif capacity is not None:
            range_ = Range(shard_id=shard_id, min_uid=rangeset.last_range().min_uid + capacity)
        else:
            raise ShardushkaError('Either `min_uid` or `capacity` is required')

        rangeset.append(range_)

    def expand_rangeset(self, rangeset_name, limit, lap_capacity, shares):
        """
        Создаёт несколько диапазонов в указанном rangeset, суммарно на limit уидов. Последний всегда открыт сверху.
        :param rangeset_name: имя rangeset'а
        :param limit: суммарное число уидов в создаваемых диапазонах
        :param lap_capacity: число уидов, за которое сделаем "полный круг" (создадим в каждом из шардов по диапазону
          и вернёмся в исходный шард)
        :param shares: доли (веса) каждого из шардов (в процентах)
        """
        rangeset = self._rangesets_by_name.get(rangeset_name)
        if rangeset is None:
            raise ShardushkaError(f'Rangeset `{rangeset_name}` not found')

        if len(shares) != len(self._shards_by_id):
            raise ShardushkaError('shares count is not equal to shards count')

        if sum(shares) != 100:
            raise ShardushkaError('shares sum is not equal to 100')

        laps_count = limit / lap_capacity
        if int(laps_count) != laps_count:
            raise ShardushkaError(f'laps_count is not integer: {limit} / {lap_capacity} = {laps_count}')
        laps_count = int(laps_count)

        capacities = [lap_capacity * share / 100 for share in shares]
        if any(int(capacity) != capacity for capacity in capacities):
            raise ShardushkaError('capacities are not integer: {}'.format(', '.join(capacities)))

        shard_ids = itertools.cycle(self._shards_by_id.keys())
        # shard_ids должны начинаться с того шарда, который следует за последним добавленным.
        # Поэтому достанем из него первые несколько элементов.
        last_shard_id = rangeset.last_range().shard_id
        for shard_id in shard_ids:
            if shard_id == last_shard_id:
                break
        shard_ids = list(itertools.islice(shard_ids, 0, len(self._shards_by_id.keys())))

        for i in range(laps_count):
            for shard_id, capacity in zip(shard_ids, capacities):
                self.create_range(
                    rangeset_name=rangeset_name,
                    shard_id=shard_id,
                    capacity=int(capacity),
                )
