# -*- coding: utf-8 -*-
from enum import Enum


class ExportFormats(Enum):
    Blackbox = 'blackbox'
    PassportPy = 'passport_py'
    PassportPl = 'passport_pl'


class Exporter:
    def __init__(self, config):
        self._config = config

    def export(self, format_):
        exporters = {
            ExportFormats.Blackbox.value: self._export_for_blackbox,
            ExportFormats.PassportPy.value: self._export_for_passport_py,
            ExportFormats.PassportPl.value: self._export_for_passport_pl,
        }
        if format_ not in exporters:
            raise NotImplementedError(f'Unknown format: {format_}')
        exporter = exporters[format_]
        return exporter()

    def _export_for_blackbox(self):
        min_uid_to_shard_id = [
            (range_.min_uid, range_.shard_id)
            for _, rangeset in self._config.ordered_rangesets
            for range_ in rangeset
        ]
        if not min_uid_to_shard_id or min_uid_to_shard_id[0][0] != 0:
            # ЧЯ проверяет, что весь диапазон uid-ов отмаплен в шарды. Поэтому лакун быть не должно.
            min_uid_to_shard_id.insert(0, (0, 1))
        return '\n'.join([
            f'{min_uid} {shard_id}'
            for min_uid, shard_id in min_uid_to_shard_id
        ])

    def _export_for_passport_py(self):
        data = [
            (range_.shard_id, range_.min_uid)
            for _, rangeset in self._config.ordered_rangesets
            for range_ in rangeset
        ]
        return repr(data)

    def _export_for_passport_pl(self):
        data = [
            f'{range_.min_uid} {range_.shard_id}'
            for _, rangeset in self._config.ordered_rangesets
            for range_ in rangeset
        ]
        return '\n'.join(data)
