# -*- coding: utf-8 -*-
from functools import wraps
import logging
import sys

from passport.backend.library.commander import Commander
from passport.backend.tools.shardushka.config import Config
from passport.backend.tools.shardushka.exceptions import ShardushkaError
from passport.backend.tools.shardushka.export import (
    Exporter,
    ExportFormats,
)


def command(read_config=True):
    def decorator(f):
        @wraps(f)
        def wrapper(config_path, *args, **kwargs):
            config = Config(config_path=config_path)
            if read_config:
                config.load()
            try:
                f(config=config, *args, **kwargs)
                config.dump()
            except ShardushkaError as e:
                logging.error(f'ERROR: {e}')
                sys.exit(1)
        return wrapper
    return decorator


# Команды утилиты -------------------------------------------------------------


@command(read_config=False)
def init(config):
    pass


@command()
def create_shard(config, shard_id):
    config.create_shard(shard_id)


@command()
def create_rangeset(config, name, shard_id, min_uid):
    config.create_rangeset(name)
    config.create_range(
        rangeset_name=name,
        shard_id=shard_id,
        min_uid=min_uid,
    )


@command()
def create_range(config, rangeset, shard_id, min_uid=None, capacity=None):
    config.create_range(
        rangeset_name=rangeset,
        shard_id=shard_id,
        min_uid=min_uid,
        capacity=capacity,
    )


@command()
def create_ranges(config, rangeset, limit, lap_capacity, shares):
    config.expand_rangeset(
        rangeset_name=rangeset,
        limit=limit,
        lap_capacity=lap_capacity,
        shares=[int(i) for i in shares.split(',')],
    )


@command()
def export(config, format_):
    print(
        Exporter(config).export(format_),
    )


# Настройка парсера аргументов -------------------------------------------------


def configure_common_args(commander):
    commander.add_argument(
        '-c', '--config',
        dest='config_path',
        default='../conf/prod.json',
    )


def configure_init_command(commander):
    commander.add_command(
        'init',
        init,
    )


def configure_create_command(commander):
    subparser = commander.add_command('create')

    subparser.add_command(
        'shard',
        create_shard,
    ).add_argument(
        '--id',
        dest='shard_id',
        type=int,
        required=True,
    )

    subparser.add_command(
        'rangeset',
        create_rangeset,
    ).add_argument(
        '--name',
        required=True,
    ).add_argument(
        '--shard',
        dest='shard_id',
        type=int,
        required=True,
    ).add_argument(
        '--min',
        dest='min_uid',
        type=int,
        required=True,
    )

    subparser.add_command(
        'range',
        create_range,
    ).add_argument(
        '--rangeset',
        required=True,
    ).add_argument(
        '--shard',
        dest='shard_id',
        type=int,
        required=True,
    ).add_argument(
        '--min',
        dest='min_uid',
        type=int,
    ).add_argument(
        '--capacity',
        type=int,
    )

    subparser.add_command(
        'ranges',
        create_ranges,
    ).add_argument(
        '--rangeset',
        required=True,
    ).add_argument(
        '--limit',
        type=int,
        required=True,
    ).add_argument(
        '--lap-capacity',
        type=int,
        required=True,
    ).add_argument(
        '--shares',
        required=True,
    )


def configure_export_command(commander):
    commander.add_command(
        'export',
        export,
    ).add_argument(
        '--format',
        dest='format_',
        choices=[item.value for item in ExportFormats],
        required=True,
    )


# -----------------------------------------------------------------------------


def main():
    logging.basicConfig(format='%(message)s', level=logging.INFO)

    commander = Commander()

    configure_common_args(commander)
    configure_init_command(commander)
    configure_create_command(commander)
    configure_export_command(commander)

    commander.invoke()
