# -*- coding: utf-8 -*-
from argparse import ArgumentParser

from passport.backend.api.settings.sharding.ranges import DB_SHARDING_RANGES
from passport.backend.core.dbmanager.sharder import build_range_shard_function


DESCRIPTION = '''
Splits input file into several files, each for one Passport shard.

Every line of input file contains space-separated `uid` and `value`. Resulting file (for some shard)
will contain all values for uids that match this shard. If `value` is omitted, `uid` will be used instead.
'''


def split_to_shards(filename, env_name, env_type):
    shards = {}
    sharder = build_range_shard_function(DB_SHARDING_RANGES[env_name][env_type])
    with open(filename, 'r') as f_in:
        for line in f_in:
            if not line:
                continue
            parts = line.strip().split(maxsplit=1)
            uid = int(parts[0])
            value = parts[1] if len(parts) > 1 else uid
            shard_id = sharder(uid)
            shards.setdefault(shard_id, []).append(value)

    for shard_id, values in shards.items():
        with open('%s.%s' % (filename, shard_id), 'w') as f_out:
            for value in values:
                f_out.write('%s\n' % value)


def main():
    parser = ArgumentParser(description=DESCRIPTION)
    parser.add_argument('filename', metavar='FILENAME', help='File with uids')
    parser.add_argument('--env_name', metavar='ENV_NAME', default='localhost')
    parser.add_argument('--env_type', metavar='ENV_TYPE', default='production')
    args = parser.parse_args()

    split_to_shards(args.filename, args.env_name, args.env_type)
