# -*- coding: utf-8 -*-
import argparse
from datetime import (
    datetime,
    timedelta,
)
import logging
import os
import random

from passport.backend.library.configurator import Configurator
from passport.backend.tools.takeout_bugreport_creator.abc_helper import ABCHelper
from passport.backend.tools.takeout_bugreport_creator.startrek_helper import (
    get_ticket_url,
    parse_datetime,
    StartekHelper,
)
from passport.backend.utils.time import unixtime_to_datetime


log = logging.getLogger('bugreport_creator')


current_user = os.environ.get('USER', os.environ.get('LOGNAME', 'root'))
DEBUG = current_user not in ('root', 'www-data')


def get_config():
    try:
        config = Configurator(
            'takeout-bugreport-creator',
            configs=[
                'settings.yaml',
                '/etc/yandex/takeout/services-production.yaml',  # конфиг тейкаута
                '~/.arcadia_settings/takeout-bugreport-creator/secrets.yaml?',
                '/usr/lib/yandex/takeout-bugreport-creator/secrets.yaml?',
            ],
        )
        if DEBUG:
            config.update(
                'settings-debug.yaml',
            )

        if not config.has('startrek', 'token'):
            raise ValueError('No Startrek token')
        if not config.has('abc', 'token'):
            raise ValueError('No ABC token')
        return config
    except (IOError, ValueError) as e:
        raise RuntimeError('Failed to load config: %s' % e)


def get_default_log_path(is_logstore=True, env_name='testing'):
    return '%(prefix)s/var/log/yandex/logbroker-client/takeout-tasks/statbox/statbox.log.%(date_part)s' % {
        'prefix': '/storage/%s' % env_name if is_logstore else '',
        'date_part': datetime.now().strftime("%Y%m%d"),
    }


def get_probable_failures(log_file_name, is_logstore=True, env_name='testing'):
    """
    Вычитывает statbox-лог Тейкаута, выбирает только записи об ошибках
    :param log_file_name: имя файла с логом
    :param is_logstore: взят ли лог с машины Тейкаута или же с логстора
    :param env_name: окружение, из которого взят лог
    :return: список словарей с полями лог-записей
    """
    if not log_file_name:
        log_file_name = get_default_log_path(is_logstore, env_name)
        log.debug('Fallback to %s for env=%s', log_file_name, env_name)
    elif log_file_name == '-':
        return {}

    # "имя машины tskv" или "tskv"
    junk_field_count = 1

    extracts_by_service_and_extract_id = {}

    config = get_config()
    min_retries_error = config['common']['min_retries_to_report']['error']
    min_retries_later = config['common']['min_retries_to_report']['later']

    with open(log_file_name, 'r') as f:
        for line in f:
            keys_and_values = line.strip().split('\t')[junk_field_count:]
            fields = dict([
                key_and_value.split('=', 1)
                for key_and_value in keys_and_values
            ])
            if not (
                fields.get('log_source') == 'logbroker' and
                fields.get('status') in ('retry_failure', 'retry_later') and
                fields.get('task_name') not in ('make_archive', 'cleanup') and
                (
                    (
                        fields.get('status') == 'retry_failure' and
                        int(fields.get('retries', 0)) >= min_retries_error
                    ) or (
                        fields.get('status') == 'retry_later' and
                        int(fields.get('retries', 0)) >= min_retries_later
                    )
                )
            ):
                # Тут всё хорошо, эта запись нам неинтересна
                continue

            if any([
                required_field_name not in fields
                for required_field_name in (
                    'service_name',
                    'extract_id',
                    'uid',
                    'retries',
                    'unixtime',
                )
            ]):
                # Какая-то кривая запись
                log.warning('Incomplete log record: %s', fields)
                continue

            service_name = fields['service_name']
            extract_id = fields['extract_id']
            retries = int(fields['retries'])
            fields['datetime'] = unixtime_to_datetime(fields['unixtime']).replace(microsecond=0)

            # для каждой пары (service_name, extract_id) храним данные последнего ретрая
            extracts_by_service_and_extract_id.setdefault(service_name, {})
            extract_retries = extracts_by_service_and_extract_id[service_name].get(extract_id, {}).get('retries')
            if extract_retries is None or retries > int(extract_retries):
                extracts_by_service_and_extract_id[service_name][extract_id] = fields

    return {
        key: list(value.values())
        for key, value in extracts_by_service_and_extract_id.items()
    }


def make_context(extract_list):
    extract_list = sorted(extract_list, key=lambda item: int(item['retries']), reverse=True)
    not_ready = [item for item in extract_list if item['status'] == 'retry_later']
    failed = [item for item in extract_list if item['status'] == 'retry_failure']
    stats = {
        'is_ok': not (not_ready or failed),
        'not_ready': {
            'count': len(not_ready),
            'extracts': not_ready,
        },
        'failed': {
            'count': len(failed),
            'extracts': failed,
        },
    }
    return stats


def get_responsibles(abc_helper, service_name):
    """Возвращает непустой упорядоченный (по убыванию ответственности) список ответственных"""
    config = get_config()

    service_config = config['services'].get(service_name)
    if service_config:
        abc_service = service_config.get('abc_service')
        if abc_service:
            responsibles = abc_helper.get_abc_members(
                abc_service=abc_service,
                abc_scope=service_config.get('abc_scope'),
            )
            if responsibles:
                # TODO: постараться отранжировать по старшинству
                random.shuffle(responsibles)
                return responsibles

        if service_config.get('responsibles'):
            return service_config['responsibles']

    return config['common']['default_responsibles']


def get_status_for_log(context, bugreport_ticket, responsibles):
    parts = []
    for env in ('test', 'prod'):
        parts.append('%s:' % env.capitalize())
        failed_count = context[env]['failed']['count']
        not_ready_count = context[env]['not_ready']['count']
        if failed_count and not_ready_count:
            parts.append('%d errors, %d long runs.' % (failed_count, not_ready_count))
        elif failed_count:
            parts.append('%d errors.' % failed_count)
        elif not_ready_count:
            parts.append('%d long runs.' % not_ready_count)
        else:
            parts.append('OK.')

    if bugreport_ticket:
        parts.append('Ticket: %s.' % get_ticket_url(bugreport_ticket.key))
    else:
        parts.append('Responsibles: %s.' % ', '.join(responsibles) or '-')

    return ' '.join(parts)


def find_tickets_to_close(service_infos, startrek_helper):
    broken_service_names = set(service_infos.keys())
    all_opened_tickets = startrek_helper.find_all_bugreport_tickets()
    tickets_to_close = [
        ticket
        for ticket in all_opened_tickets
        if set(ticket.tags).isdisjoint(broken_service_names)
    ]
    return tickets_to_close


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-t', '--test_log')
    parser.add_argument('-p', '--prod_log')
    parser.add_argument('-y', '--force_yes', action='store_true', help='Create tickets automatically')
    parser.add_argument('--ignore_testing', action='store_true', help='Don\'t create ticket if prod is OK')
    parser.add_argument('--notify_disabled', action='store_true', help='Create tickets even for disabled services')
    return parser.parse_args()


def run():
    try:
        args = parse_args()

        config = get_config()
        config.set_logging()

        startrek_helper = StartekHelper(
            oauth_token=config['startrek']['token'],
            queue=config['startrek']['queue'],
        )
        abc_helper = ABCHelper(
            oauth_token=config['abc']['token'],
        )

        test_data = get_probable_failures(args.test_log, env_name='testing')
        prod_data = get_probable_failures(args.prod_log, env_name='production')

        service_infos = {}
        if args.ignore_testing:
            service_names = sorted(prod_data.keys())
        else:
            service_names = sorted(set(list(test_data.keys()) + list(prod_data.keys())))
        for service_name in service_names:
            service_config = config['services'].get(service_name)
            if not (args.notify_disabled or service_config and service_config.get('enabled')):
                continue  # сервис не включен в граф, не нужно его пинать

            context = {
                'dt': datetime.now().replace(microsecond=0),
                'service_name': service_name,
                'test': make_context(test_data.get(service_name, [])),
                'prod': make_context(prod_data.get(service_name, [])),
            }
            bugreport_ticket = startrek_helper.find_recent_bugreport_ticket(
                service_name,
                reopen_interval=config['common']['reopen_interval'],
            )

            responsibles = []
            if bugreport_ticket is None:
                responsibles = get_responsibles(abc_helper, service_name)
                max_count = config['common']['max_responsibles_count']
                if len(responsibles) > max_count:
                    log.warning(
                        '[%s] Too many responsibles found: %s. Using first %s.',
                        service_name,
                        len(responsibles),
                        max_count,
                    )
                    responsibles = responsibles[:max_count]

            log.info('[%s] %s', service_name, get_status_for_log(context, bugreport_ticket, responsibles))

            service_infos[service_name] = {
                'context': context,
                'bugreport_ticket': bugreport_ticket,
                'responsibles': responsibles,
            }

        if not service_infos:
            log.info('Wow! Everything is working like a charm.')

        tickets_to_close = find_tickets_to_close(service_infos=service_infos, startrek_helper=startrek_helper)
        if tickets_to_close:
            log.info(
                'Some services have fixed themselves. Ready to close: %s.' % ', '.join([
                    get_ticket_url(ticket.key)
                    for ticket in tickets_to_close
                ]),
            )
            if args.force_yes or input('Close old ST tickets? (y/N) ').strip().lower() == 'y':
                for ticket in tickets_to_close:
                    startrek_helper.post_status_as_comment(
                        ticket=ticket,
                        context={},
                        is_ok=True,
                    )
                    startrek_helper.close_ticket(ticket)

        if service_infos and (args.force_yes or input('Make new ST tickets? (y/N) ').strip().lower() == 'y'):
            deadline = datetime.now() + timedelta(seconds=config['common']['deadline'])

            for service_name, service_info in service_infos.items():
                bugreport_ticket = service_info['bugreport_ticket']
                if bugreport_ticket is None:
                    ticket_key = startrek_helper.create_ticket(
                        context=service_info['context'],
                        assignee=service_info['responsibles'][0],
                        followers=service_info['responsibles'][1:],
                        deadline=deadline,
                    )
                    log.info('[%s] Created %s', service_name, get_ticket_url(ticket_key))
                else:
                    ticket_url = get_ticket_url(bugreport_ticket.key)
                    if datetime.now() - parse_datetime(bugreport_ticket.updatedAt) < timedelta(seconds=config['common']['min_ping_interval']):
                        log.info('[%s] Nothing to do with %s: it was updated recently', service_name, ticket_url)
                    else:
                        if bugreport_ticket.status.key == 'closed':
                            startrek_helper.open_ticket(bugreport_ticket)

                        startrek_helper.post_status_as_comment(
                            ticket=bugreport_ticket,
                            context=service_info['context'],
                            is_ok=False,
                        )
                        log.info('[%s] Updated %s', service_name, ticket_url)

    except Exception as e:
        log.exception('ERROR: %s(%s)', e.__class__.__name__, e.__dict__)
        raise
