# -*- coding: utf-8 -*-
from datetime import (
    datetime,
    timedelta,
)
import os

from jinja2 import Environment
from passport.backend.utils.template_loader import CachedTemplateLoader
from startrek_client import Startrek


COMMON_ST_TAG = u'takeout_проблемы_сервисов'


def render_template(context, template_name):
    template_dir = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'templates')
    environment = Environment(
        loader=CachedTemplateLoader(template_dir),
        trim_blocks=True,
        lstrip_blocks=True,
        keep_trailing_newline=True,
    )
    template = environment.get_template(template_name)

    return template.render(**context)


def parse_datetime(dt_string):
    return datetime.strptime(
        dt_string.split('.')[0],
        '%Y-%m-%dT%H:%M:%S',
    )


def get_ticket_url(ticket_key):
    return 'https://st.yandex-team.ru/%s' % ticket_key


class StartekHelper(object):
    def __init__(self, oauth_token, queue):
        self.startrek_client = Startrek(
            useragent='Takeout Bugreport Creator',
            base_url='https://st-api.yandex-team.ru',
            token=oauth_token,
        )
        self.queue = queue

    def find_all_bugreport_tickets(self):
        tickets = self.startrek_client.issues.find(
            query=(
                'Queue: "%s" AND '
                'Tags: "%s" AND '
                'Resolution: empty()'
            ) % (
                self.queue,
                COMMON_ST_TAG,
            )
        )
        return sorted(tickets, key=lambda x: x.key)

    def find_recent_bugreport_ticket(self, service_name, reopen_interval):
        cutoff = datetime.now() - timedelta(seconds=reopen_interval)  # чтобы не создавать новый тикет через час после закрытия старого

        possible_tickets = self.startrek_client.issues.find(
            query=(
                'Queue: "%s" AND '
                'Tags: "%s" AND '
                'Tags: "%s" AND '
                '(Resolution: empty() OR (Resolution: !empty() AND Resolved: >= %s))'
            ) % (
                self.queue,
                COMMON_ST_TAG,
                service_name,
                cutoff.strftime("%Y-%m-%d"),
            )
        )
        possible_tickets = sorted(possible_tickets, key=lambda x: parse_datetime(x.updatedAt))
        return possible_tickets[0] if possible_tickets else None

    def create_ticket(self, context, assignee, followers=None, deadline=None):
        service_name = context['service_name']
        response = self.startrek_client.issues.create(
            assignee=assignee,
            followers=followers,
            queue=self.queue,
            summary=u'Сервис %s: проблемы с выгрузками Takeout' % service_name,
            description=render_template(context, template_name='ticket_body.j2'),
            tags=[COMMON_ST_TAG, service_name],
            deadline=deadline.strftime('%Y-%m-%d'),
        )
        return response.key

    @staticmethod
    def post_status_as_comment(ticket, context, is_ok):
        if is_ok:
            template_name = 'ticket_comment_ok.j2'
        else:
            template_name = 'ticket_comment_error.j2'

        ticket.comments.create(
            text=render_template(context, template_name=template_name),
        )
        return ticket.key

    @staticmethod
    def open_ticket(ticket):
        ticket.transitions['reopen'].execute()
        return ticket.key

    @staticmethod
    def close_ticket(ticket):
        ticket.transitions['close'].execute(resolution='fixed')
        return ticket.key
