import json
import logging
import socket
from time import time
from typing import List

from passport.backend.core.logging_utils.helpers import mask_sensitive_fields
from passport.backend.tvm_keyring import settings
from passport.backend.tvm_keyring.exceptions import (
    TVMPermanentError,
    TVMTemporaryError,
)
import requests
import ticket_parser2


log = logging.getLogger('tvm')


class TVM(object):
    # TODO: перейти на passport.backend.core.builders.tvm, когда он выделится в отдельный ya.make
    def __init__(self):
        self._url = settings.TVM_URL
        self._timeout = settings.TVM_TIMEOUT
        self._retries = settings.TVM_RETRIES

    def _request(self, **kwargs):
        return requests.request(**kwargs)

    def _make_request(self, method, url_suffix, params=None, data=None):
        response = None
        url = '%s/%s' % (self._url, url_suffix)
        for _ in range(self._retries):
            try:
                log.debug('Requesting TVM: %s with %s', url, mask_sensitive_fields(params or data))
                response = self._request(
                    method=method,
                    url=url,
                    data=data,
                    params=params,
                    timeout=self._timeout,
                    verify=settings.SSL_CA_CERT,
                )
                if response.status_code >= 500:
                    log.warning('TVM responded with status=%s', response.status_code)
                    continue
                else:
                    break
            except (requests.Timeout, requests.ConnectionError, socket.error) as e:
                log.warning('Error requesting %s', url, exc_info=e)

        if response is None:
            raise TVMTemporaryError('TVM request failed')
        elif response.status_code not in (200, 400):
            raise TVMPermanentError('TVM responded with status=%s', response.status_code)
        else:
            return response

    def client_credentials(self, src: int, dst: List[int], service_context):
        ts = int(time())
        resp = self._make_request(
            method='POST',
            url_suffix='2/ticket',
            data=dict(
                grant_type='client_credentials',
                src=src,
                dst=','.join(map(str, dst)),
                ts=ts,
                sign=service_context.sign(ts, dst),
            ),
        )
        rv = json.loads(resp.content)
        if resp.status_code == 200:
            return rv
        else:
            raise TVMPermanentError('TVM responded with error: %s' % rv['desc'])

    def keys(self) -> str:
        keys_bytes = self._make_request(
            method='GET',
            url_suffix='2/keys',
            params={'lib_version': ticket_parser2.__version__},
        ).content
        # ServiceContext принимает строки и сам декодирует их в байты, поэтому надо привести байты к строке
        return keys_bytes.decode('utf-8')
