import json
import logging
import os

from atomicwrites import atomic_write
from passport.backend.tvm_keyring import settings
from passport.backend.tvm_keyring.exceptions import (
    BaseError,
    BaseTVMError,
    ConfigInvalidError,
    ResultInvalidError,
)
from passport.backend.tvm_keyring.tvm import TVM
from passport.backend.tvm_keyring.utils import (
    change_file_owner,
    get_config_filename,
    get_config_names,
    get_current_user,
    get_groups_of_current_user,
    get_keys_filename,
    get_result_filename,
    is_up_to_date,
    parse_and_validate_config,
)
from ticket_parser2 import BlackboxEnv
from ticket_parser2.exceptions import ContextException
from ticket_parser2.low_level import (
    ServiceContext,
    UserContext,
)


log = logging.getLogger('tvm_keyring')


def _dump_result(filename, data, owner, group, permissions):
    with atomic_write(filename, overwrite=True) as f:
        f.write(data)
    os.chmod(filename, permissions)
    change_file_owner(filename, owner, group)


def _try_update_keys():
    result_filename = get_keys_filename()

    if is_up_to_date(result_filename, settings.KEYS_UPDATE_INTERVAL):
        log.debug('Keys are already up-to-date')
        keys = None
    else:
        try:
            keys = TVM().keys()
            # Попробуем создать контексты, чтобы убедиться в валидности скачанных ключей
            ServiceContext(
                client_id=settings.FAKE_CLIENT_ID,
                secret=settings.FAKE_CLIENT_SECRET,
                tvm_keys=keys,
            )
            for env in BlackboxEnv:
                UserContext(
                    env=env,
                    tvm_keys=keys,
                )
        except BaseTVMError as e:
            log.warning(e)
            keys = None
        except ContextException as e:
            log.warning('Failed to create context with downloaded keys: %s', e)
            keys = None

    if keys is not None:
        _dump_result(
            result_filename,
            keys,
            owner=settings.RESULT_KEYS_OWNER,
            group=settings.RESULT_KEYS_GROUP,
            permissions=settings.RESULT_KEYS_PERMISSIONS,
        )
    elif os.path.exists(result_filename):
        with open(result_filename, 'r') as f:
            keys = f.read()
    else:
        raise ResultInvalidError('No TVM keys available')

    return keys


def _try_update_tickets(tvm_keys, force=False):
    config_names = get_config_names()

    # Удаляем лишнее в каталоге с результатами
    for result_name in sorted(os.listdir(settings.RESULT_PATH)):
        if result_name.replace('.tickets', '') not in config_names and result_name != 'tvm.keys':
            result_path = os.path.join(settings.RESULT_PATH, result_name)
            log.debug('Deleting unused %s...', result_path)
            try:
                os.remove(result_path)
            except OSError:
                log.warning('Unable to delete %s', result_path)

    # Пытаемся переполучить тикеты, если уже пора
    for config_name in config_names:
        log.debug('Getting tickets for %s...', config_name)
        config_filename = get_config_filename(config_name)
        result_filename = get_result_filename(config_name)
        if is_up_to_date(result_filename, settings.TICKETS_UPDATE_INTERVAL, config_filename=config_filename) and not force:
            log.debug('[%s] Tickets are already up-to-date', config_name)
            continue

        try:
            config_data = parse_and_validate_config(
                config_name,
                current_user=get_current_user(),
                groups_of_current_user=get_groups_of_current_user(),
            )
        except ConfigInvalidError as e:
            log.warning('[%s] Invalid config: %s', config_name, e)
            continue

        service_context = ServiceContext(
            client_id=config_data['client_id'],
            secret=config_data.get('client_secret'),
            tvm_keys=tvm_keys,  # ключи мы уже провалидировали выше, контекст обязан создаться успешно
        )
        try:
            if config_data.get('destinations'):
                rv = TVM().client_credentials(
                    src=config_data['client_id'],
                    dst=[dst['client_id'] for dst in config_data['destinations']],
                    service_context=service_context,
                )
                client_id_to_alias = {dst['client_id']: dst['alias'] for dst in config_data['destinations']}
                for client_id, destination in rv.items():
                    rv[client_id].update(
                        alias=client_id_to_alias[int(client_id)],
                    )
            else:
                rv = {}
            result_data = dict(
                client_id=config_data['client_id'],
                client_secret=config_data.get('client_secret'),
                tickets=rv,
            )

            result_properties = config_data.get('result', {})
            result_permissions = settings.RESULT_TICKETS_DEFAULT_PERMISSIONS
            if result_properties.get('permissions'):
                result_permissions = int(result_properties['permissions'], 8)

            _dump_result(
                result_filename,
                json.dumps(result_data, indent=2),
                owner=result_properties.get('owner', settings.RESULT_TICKETS_DEFAULT_OWNER),
                group=result_properties.get('group', settings.RESULT_TICKETS_DEFAULT_GROUP),
                permissions=result_permissions,
            )

        except BaseTVMError as e:
            log.warning('[%s] %s', config_name, e)


def update(force=False):
    log.info('Task started')
    try:
        keys = _try_update_keys()
        _try_update_tickets(keys, force=force)
    except BaseError as e:
        log.error(e)
        return False
    except Exception as e:
        log.error('Unhandled error: %s', e, exc_info=True)
        return False
    else:
        log.info('Task completed')
        return True
