from collections import Counter
import getpass
import grp
import json
import os
import pwd
from time import time

import jsonschema
from passport.backend.tvm_keyring import settings
from passport.backend.tvm_keyring.exceptions import ConfigInvalidError


CONFIG_JSON_SCHEMA = {
    '$schema': 'http://json-schema.org/schema#',
    'type': 'object',
    'properties': {
        'client_id': {'type': 'integer'},
        'client_secret': {'type': 'string', 'minLength': 22, 'maxLength': 22},
        'destinations': {
            'type': 'array',
            'uniqueItems': True,
            'items': {
                'type': 'object',
                'properties': {
                    'client_id': {'type': 'integer'},
                    'scopes': {
                        'type': 'array',
                        'minItems': 1,
                        'uniqueItems': True,
                        'items': {'type': 'string', 'minLength': 1},
                    },
                    'alias': {'type': 'string', 'minLength': 1},
                },
                'required': ['client_id', 'alias'],
                'additionalProperties': False,
            },
        },
        'result': {
            'type': 'object',
            'properties': {
                'owner': {'type': 'string', 'minLength': 1},
                'group': {'type': 'string', 'minLength': 1},
                'permissions': {'type': 'string',  "pattern": "^[0-7]{3}$"},
            },
            'additionalProperties': False,
        },
    },
    'required': ['client_id'],
    'additionalProperties': False,
}

SECRET_CONFIG_JSON_SCHEMA = {
    '$schema': 'http://json-schema.org/schema#',
    'type': 'object',
    'properties': {
        'client_id': {'type': 'integer'},
        'client_secret': {'type': 'string', 'minLength': 22, 'maxLength': 22},
    },
    'required': ['client_id', 'client_secret'],
    'additionalProperties': False,
}


def _read_file_and_validate_schema(filename, schema):
    with open(filename, 'r') as f:
        data = json.load(f)
    jsonschema.validate(data, schema)
    return data


def parse_and_validate_config(config_name, current_user, groups_of_current_user):
    config_filename = os.path.join(settings.CONFIG_PATH, config_name)
    try:
        data = _read_file_and_validate_schema(config_filename, CONFIG_JSON_SCHEMA)
        aliases = [dst['alias'] for dst in data.get('destinations', [])]
        duplicate_aliases = [alias for alias, count in Counter(aliases).items() if count > 1]
        if duplicate_aliases:
            raise ConfigInvalidError('[%s] Duplicate aliases: %s' % (config_name, ', '.join(duplicate_aliases)))

        is_secret_required = bool(data.get('destinations'))
        is_secret_provided = bool(data.get('client_secret'))
        if is_secret_required and not is_secret_provided:
            # попробуем найти секрет в отдельном файле
            secret_config_filename = get_secret_config_filename(config_name)
            if os.path.isfile(secret_config_filename):
                secret_data = _read_file_and_validate_schema(secret_config_filename, SECRET_CONFIG_JSON_SCHEMA)
                if data['client_id'] != secret_data['client_id']:
                    raise ConfigInvalidError(
                        '[%s] Invalid config: \'client_id\' from %s and %s do not match' % (
                            config_name,
                            config_filename,
                            secret_config_filename,
                        ),
                    )
                data.update(secret_data)
                is_secret_provided = True

        if is_secret_required and not is_secret_provided:
            raise ConfigInvalidError(
                '[%s] Invalid config: either \'client_secret\' or extra file with secret is required when destinations are present' % (
                    config_name,
                ),
            )

        if data.get('result') and current_user != 'root':
            if data['result'].get('owner') and data['result']['owner'] != current_user:
                raise ConfigInvalidError(
                    '[%s] Insufficient rights: unable to chown to \'%s\' while running as `%s`' % (
                        config_name,
                        data['result']['owner'],
                        current_user,
                    ),
                )

            if data['result'].get('group') and data['result']['group'] not in groups_of_current_user:
                raise ConfigInvalidError(
                    '[%s] Insufficient rights: unable to chgrp to \'%s\' while running as `%s`' % (
                        config_name,
                        data['result']['group'],
                        current_user,
                    ),
                )

        return data
    except IOError as e:
        raise ConfigInvalidError('[%s] Cannot read config: %s' % (config_name, e))
    except json.decoder.JSONDecodeError as e:
        raise ConfigInvalidError('[%s] Bad JSON in config: %s' % (config_name, e))
    except jsonschema.ValidationError as e:
        raise ConfigInvalidError('[%s] Invalid config: %s' % (config_name, e.message))


def _get_file_age(filename):
    if not (filename and os.path.exists(filename)):
        mtime = 0
    else:
        mtime = os.path.getmtime(filename)
    return time() - mtime


def is_up_to_date(filename, max_age, config_filename=None):
    file_age = _get_file_age(filename)
    config_age = _get_file_age(config_filename)
    return file_age <= max_age and file_age <= config_age


def get_config_names():
    return sorted(set(os.listdir(settings.CONFIG_PATH)))


def get_config_filename(config_name):
    return os.path.join(settings.CONFIG_PATH, config_name)


def get_secret_config_filename(config_name):
    return os.path.join(settings.SECRET_CONFIG_PATH, config_name)


def get_result_filename(config_name):
    return os.path.join(settings.RESULT_PATH, '%s.tickets' % config_name)


def get_keys_filename():
    return os.path.join(settings.RESULT_PATH, 'tvm.keys')


def change_file_owner(filename, user, group):
    uid = pwd.getpwnam(user).pw_uid
    gid = grp.getgrnam(group).gr_gid
    os.chown(filename, uid, gid)


def get_current_user():
    return getpass.getuser()


def get_groups_of_current_user():
    return [
        grp.getgrgid(g).gr_name
        for g in os.getgroups()
    ]
