import json
import logging

from passport.backend.tvm_keyring import settings
from passport.backend.tvm_keyring.exceptions import (
    BaseError,
    ResultExpiresSoonError,
    ResultIncompleteError,
    ResultInvalidError,
    ResultOutdatedError,
)
from passport.backend.tvm_keyring.utils import (
    get_config_names,
    get_current_user,
    get_groups_of_current_user,
    get_keys_filename,
    get_result_filename,
    is_up_to_date,
    parse_and_validate_config,
)
from ticket_parser2.exceptions import ContextException
from ticket_parser2.low_level import ServiceContext


log = logging.getLogger('tvm_keyring')


STATUS_OK = 0
STATUS_WARNING = 1
STATUS_CRITICAL = 2

MESSAGE_OK = 'OK'


def _validate_keys():
    keys_filename = get_keys_filename()
    try:
        with open(keys_filename, 'r') as f:
            keys = f.read()
        ServiceContext(
            client_id=settings.FAKE_CLIENT_ID,
            secret=settings.FAKE_CLIENT_SECRET,
            tvm_keys=keys,
        )
    except IOError as e:
        raise ResultInvalidError('Cannot read keys: %s' % e)
    except ContextException as e:
        raise ResultInvalidError('Failed to create context with downloaded keys: %s' % e)

    if not is_up_to_date(keys_filename, settings.MAX_KEYS_AGE_CRIT):
        raise ResultOutdatedError('Keys outdated')
    elif not is_up_to_date(keys_filename, settings.MAX_KEYS_AGE_WARN):
        raise ResultExpiresSoonError('Keys are to expire soon')


def _validate_tickets():
    config_names = get_config_names()

    # Валидируем конфиги
    for config_name in config_names:
        parse_and_validate_config(
            config_name,
            current_user=get_current_user(),
            groups_of_current_user=get_groups_of_current_user(),
        )

    # Проверяем тикеты
    for config_name in config_names:
        result_filename = get_result_filename(config_name)
        # проверим состав файла с результатом
        try:
            with open(result_filename, 'r') as f:
                data = json.load(f)
                if not data.get('client_id') or (data.get('tickets') and not data.get('client_secret')):
                    raise ResultInvalidError('[%s] Client credentials missing in result' % config_name)
                for dst, dst_data in data.get('tickets', {}).items():
                    if not dst_data.get('ticket'):
                        raise ResultIncompleteError(
                            '[%s] Ticket missing for dst=%s: %s' % (config_name, dst, dst_data.get('error')),
                        )
        except IOError as e:
            raise ResultInvalidError('[%s] Cannot read tickets: %s' % (config_name, e))
        except json.decoder.JSONDecodeError as e:
            raise ResultInvalidError('[%s] Bad JSON in tickets: %s' % (config_name, e))

        # проверим свежесть файла
        if not is_up_to_date(result_filename, settings.MAX_TICKETS_AGE_CRIT):
            raise ResultOutdatedError('[%s] Tickets outdated' % config_name)
        elif not is_up_to_date(result_filename, settings.MAX_TICKETS_AGE_WARN):
            raise ResultExpiresSoonError('[%s] Tickets are to expire soon' % config_name)


def validate():
    try:
        _validate_keys()
        _validate_tickets()
    except ResultExpiresSoonError as e:
        status, message = STATUS_WARNING, str(e)
        log.warning('Validation failed: %s', message)
    except BaseError as e:
        status, message = STATUS_CRITICAL, str(e)
        log.error('Validation failed: %s', message)
    except Exception as e:
        status, message = STATUS_CRITICAL, 'Unhandled error: %s' % e
        log.error(message, exc_info=True)
    else:
        status, message = STATUS_OK, MESSAGE_OK

    return '%s;%s' % (status, message)
