package uaasproxy

import (
	"context"
	"encoding/base64"
	"errors"
	"fmt"
	"net/http"
	"strconv"
	"strings"

	"github.com/mailru/easyjson"
)

var HeaderExpBoxes = "X-Yandex-Expboxes"
var HeaderExpFlags = "X-Yandex-Expflags"

type Experiment struct {
	Handler string
	TestID  int
	Context UaasContext
}

type ExperimentData struct {
	Experiments []Experiment
}

//easyjson:json
type UaasResponseExperiments []UaasResponseExperiment

//easyjson:json
type UaasResponseExperiment struct {
	Handler string      `json:"HANDLER"`
	Context UaasContext `json:"CONTEXT"`
}

//easyjson:json
type UaasContext struct {
	Passport UaasFlags `json:"PASSPORT"`
}

//easyjson:json
type UaasFlags struct {
	Flags []string `json:"flags"`
}

type UaasEnvData struct {
	UserIP     string
	SplitKey   string
	SplitValue string
	TestIds    []int
}

func (t *UaasProxy) QueryUaas(ctx context.Context, httpAPIClient *HTTPApiClient, configName string, env UaasEnvData) (ExperimentData, error) {
	queryURL := fmt.Sprintf("%s/%s", httpAPIClient.BaseURL, configName)

	request, err := http.NewRequestWithContext(ctx, "GET", queryURL, nil)
	if err != nil {
		return ExperimentData{}, err
	}
	headers := request.Header
	query := request.URL.Query()

	if len(env.SplitKey) > 0 && len(env.SplitValue) > 0 {
		query.Add(env.SplitKey, env.SplitValue)
	}
	// запрос надо отправить в тестинг
	if len(env.TestIds) > 0 {
		headers.Set("X-Yandex-UAAS", "testing")
		testIDs := strings.Join(arrayItoA(env.TestIds), "_")
		query.Add("test-id", testIDs)
	} else if len(env.UserIP) > 0 {
		// real_ip не участвует в запросе в testing,
		// т.к. иначе не выдаются запрошенные в test_ids выборки
		headers.Set("X-Forwarded-For-Y", env.UserIP)
	}
	request.URL.RawQuery = query.Encode()

	response, err := httpAPIClient.Client.Do(request)
	if err != nil {
		return ExperimentData{}, err
	}
	if response != nil {
		defer response.Body.Close()
	}
	if response.StatusCode != http.StatusOK {
		return ExperimentData{}, fmt.Errorf(
			"got %d status code from uaas, expected %d",
			response.StatusCode,
			http.StatusOK,
		)
	}

	return t.parseUaasResponse(response)
}

func parseUaasBoxes(boxesHeader string) ([]int, error) {
	var testIds []int
	if len(boxesHeader) == 0 {
		return testIds, nil
	}

	boxes := strings.Split(boxesHeader, ";")
	for _, box := range boxes {
		n := strings.Index(box, ",")
		if n == -1 {
			return testIds, errors.New("malformed box, missing comma")
		}
		testID := box[:n]
		digit, err := strconv.Atoi(testID)
		if err != nil {
			return testIds, err
		}
		testIds = append(testIds, digit)
	}
	return testIds, nil
}

func (t *UaasProxy) parseUaasSingleFlagPool(singleFlag string) (UaasResponseExperiments, error) {
	fc := t.pools.flagsPool.Get().(map[string]UaasResponseExperiments)
	defer t.pools.flagsPool.Put(fc)
	cached, ok := fc[singleFlag]
	if ok {
		return cached, nil
	}

	var parsedJSON UaasResponseExperiments
	rawJSON := make([]byte, base64.StdEncoding.DecodedLen(len(singleFlag)))
	n, err := base64.StdEncoding.Decode(rawJSON, []byte(singleFlag))

	if err != nil {
		return nil, err
	}
	if err := easyjson.Unmarshal(rawJSON[:n], &parsedJSON); err != nil {
		return nil, err
	}
	fc[singleFlag] = parsedJSON
	return parsedJSON, nil
}

func (t *UaasProxy) parseUaasFlags(boxes []int, flagsHeader string) ([]Experiment, error) {
	var experiments []Experiment

	if len(flagsHeader) == 0 {
		return experiments, nil
	}

	blobs := strings.Split(flagsHeader, ",")
	if len(boxes) != len(blobs) {
		return nil, fmt.Errorf("boxes and flags count mismatch: %d != %d", len(boxes), len(blobs))
	}
	for i, blob := range blobs {
		parsedJSON, err := t.parseUaasSingleFlagPool(blob)
		if err != nil {
			return nil, err
		}
		for _, context := range parsedJSON {
			/* пропускаем любые эксперименты, кроме паспортных */
			if context.Handler == "PASSPORT" {
				experiments = append(
					experiments,
					Experiment{
						context.Handler,
						boxes[i],
						context.Context,
					},
				)
			}
		}
	}

	return experiments, nil
}

func (t *UaasProxy) parseUaasResponse(r *http.Response) (ExperimentData, error) {
	ed := t.pools.experimentsPool.Get().(ExperimentData)
	defer t.pools.experimentsPool.Put(ed)
	ed.Experiments = ed.Experiments[:0]

	boxesHeader := getHeaderValue(r.Header[HeaderExpBoxes])
	flagsHeader := getHeaderValue(r.Header[HeaderExpFlags])

	boxes, err := parseUaasBoxes(boxesHeader)
	if err != nil {
		return ed, err
	}

	experiments, err := t.parseUaasFlags(boxes, flagsHeader)
	if err != nil {
		return ed, err
	}

	ed.Experiments = experiments
	return ed, nil
}
