package uaasproxy

import (
	_ "embed"
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/passport/backend/uaas_proxy/internal/translations"
)

//go:embed divcards/default_rationale.json
var divcardDefaultRationale string

//go:embed divcards/default_blocked_rationale.json
var divcardDefaultBlockedRationale string

type Params struct {
	Language     string
	Color        string
	TemplateName string
}

func getPalette() map[string]map[string]string {
	palette := map[string]map[string]string{}
	palette["dark"] = map[string]string{
		"passport_contacts_close_icon": "drawable:///android/passport_contacts_close_icon?dark",
		"background_color":             "#000000",
		"placeholder_color":            "#00000000",
		"text_color":                   "#ffffff",
	}
	palette["light"] = map[string]string{
		"passport_contacts_close_icon": "drawable:///android/passport_contacts_close_icon?light",
		"background_color":             "#ffffff",
		"placeholder_color":            "#00ffffff",
		"text_color":                   "#000000",
	}
	return palette
}

func cardToTemplate(card string) (string, error) {
	if card == "default_rationale" {
		return card, nil
	} else if card == "default_blocked_rationale" {
		return card, nil
	} else {
		return "", fmt.Errorf("unexpected card name %s", card)
	}
}

func parseDivcardForm(r *http.Request) (Params, error) {
	err := r.ParseForm()
	if err != nil {
		return Params{}, err
	}
	lang := strings.Trim(r.Form.Get("lang"), " ")
	if len(lang) == 0 {
		return Params{}, fmt.Errorf("lang.empty")
	}
	color := strings.Trim(r.Form.Get("color"), " ")
	if len(color) == 0 {
		return Params{}, fmt.Errorf("color.empty")
	}
	divcard := strings.Trim(r.Form.Get("card"), " ")
	template, err := cardToTemplate(divcard)
	if err != nil {
		return Params{}, fmt.Errorf("card.invalid")
	}
	return Params{Language: lang, Color: color, TemplateName: template}, nil
}

func renderTemplate(tr translations.TranslationsCache, templateName string, lang string, color string) (string, error) {
	var retval string
	var tmp string
	var err error
	var translationKeys []string
	if templateName == "default_rationale" {
		retval = divcardDefaultRationale
		translationKeys = []string{
			"permission_request_first_shown_ok_button",
			"permission_request_first_shown_cancel_button",
			"permission_request_first_shown_title",
			"permission_request_first_shown_description",
		}
	} else if templateName == "default_blocked_rationale" {
		retval = divcardDefaultBlockedRationale
		translationKeys = []string{
			"permission_request_rerun_button",
			"permission_request_rerun_description",
		}
	}
	for _, key := range translationKeys {
		tmp = strings.Replace(retval, key, tr.Get(lang, key), 1)
		if tmp == retval {
			panic(fmt.Errorf("replacing key %s didnt change anything in template %s", key, templateName))
		}
		retval = tmp
	}
	palette := getPalette()
	paletteKeys := []string{
		"passport_contacts_close_icon",
		"background_color",
		"placeholder_color",
		"text_color",
	}
	for _, key := range paletteKeys {
		tmp = strings.Replace(retval, key, palette[color][key], 1)
		if tmp == retval {
			panic(fmt.Errorf("replacing key %s didnt change anything in template %s", key, templateName))
		}
		retval = tmp
	}

	return retval, err
}

func (t *UaasProxy) increaseDivcardsMetrics(params Params) error {
	if params.Language == "ru" && params.Color == "light" && params.TemplateName == "default_rationale" {
		t.unistat.divRuLightDefaultRationale.Inc()
	} else if params.Language == "en" && params.Color == "light" && params.TemplateName == "default_rationale" {
		t.unistat.divEnLightDefaultRationale.Inc()
	} else if params.Language == "ru" && params.Color == "dark" && params.TemplateName == "default_rationale" {
		t.unistat.divRuDarkDefaultRationale.Inc()
	} else if params.Language == "en" && params.Color == "dark" && params.TemplateName == "default_rationale" {
		t.unistat.divEnDarkDefaultRationale.Inc()
	} else /* next card */ if params.Language == "ru" && params.Color == "light" && params.TemplateName == "default_blocked_rationale" {
		t.unistat.divRuLightDefaultBlockedRationale.Inc()
	} else if params.Language == "en" && params.Color == "light" && params.TemplateName == "default_blocked_rationale" {
		t.unistat.divEnLightDefaultBlockedRationale.Inc()
	} else if params.Language == "ru" && params.Color == "dark" && params.TemplateName == "default_blocked_rationale" {
		t.unistat.divRuDarkDefaultBlockedRationale.Inc()
	} else if params.Language == "en" && params.Color == "dark" && params.TemplateName == "default_blocked_rationale" {
		t.unistat.divEnDarkDefaultBlockedRationale.Inc()
	} else {
		return fmt.Errorf("unexpected combination of params: %s, %s, %s", params.Language, params.Color, params.TemplateName)
	}

	return nil
}

func (t *UaasProxy) HandlerDivcard() echo.HandlerFunc {
	return func(c echo.Context) error {
		t.unistat.divTotal.Inc()
		before := time.Now()
		params, err := parseDivcardForm(c.Request())
		if err != nil {
			t.unistat.divFailed.Inc()
			return t.respondWithError(c, err.Error())
		}
		err = t.increaseDivcardsMetrics(params)
		if err != nil {
			t.unistat.divFailed.Inc()
			t.logger.Warnf("Failed to increase yasm metrics: %s", err.Error())
		}
		retval, err := renderTemplate(t.translations, params.TemplateName, params.Language, params.Color)
		if err != nil {
			t.unistat.divFailed.Inc()
			return t.respondWithError(c, "failed to encode response")
		}
		duration := float64(time.Since(before) / time.Millisecond)
		t.unistat.divDuration.Inc(duration)
		t.unistat.divSuccess.Inc()
		c.Response().Header().Set(echo.HeaderContentType, "application/json")
		return c.String(http.StatusOK, retval)
	}
}
