package uaasproxy

import (
	"context"
	"errors"
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/labstack/echo/v4"
	"github.com/mailru/easyjson"
)

//easyjson:json
type ErrorResponse struct {
	Status string   `json:"status"`
	Errors []string `json:"errors"`
}

//easyjson:json
type SuccessResponse struct {
	Status      string               `json:"status"`
	Experiments []ResponseExperiment `json:"experiments"`
}

//easyjson:json
type ResponseExperiment struct {
	Passport UaasFlags `json:"PASSPORT"`
	TestID   int       `json:"test_id"`
	Handler  string    `json:"handler"`
}

func (t *UaasProxy) respondWithError(c echo.Context, errorText string) error {
	responseStruct := ErrorResponse{
		Status: "error",
		Errors: []string{errorText},
	}

	responseJSON, err := easyjson.Marshal(responseStruct)
	if err != nil {
		panic(err)
	}
	c.Response().Header().Set(echo.HeaderContentType, "application/json")
	return c.String(http.StatusOK, string(responseJSON))
}

func getRealIP(r *http.Request) string {
	realIP := strings.Trim(r.Form.Get("real_ip"), " ")
	if len(realIP) == 0 {
		realIP = strings.Trim(getHeaderValue(r.Header["X-Yproxy-Header-Ip"]), " ")
		if len(realIP) == 0 {
			realIP = strings.Trim(getHeaderValue(r.Header["Ya-Consumer-Client-Ip"]), " ")
		}
	}
	return realIP
}

func parseForm(r *http.Request) (UaasEnvData, error) {
	err := r.ParseForm()
	if err != nil {
		return UaasEnvData{}, fmt.Errorf("form.invalid")
	}
	rawTestIDs := strings.Trim(r.Form.Get("test_ids"), " ")
	testIDs := make([]int, len(rawTestIDs))
	if len(rawTestIDs) > 0 {
		testIDs, err = arrayAtoI(
			strings.Split(
				rawTestIDs,
				",",
			),
		)
		if err != nil {
			return UaasEnvData{}, fmt.Errorf("test_ids.invalid")
		}
	}

	realIP := getRealIP(r)

	if len(realIP) > 0 && !isIPAddress(realIP) {
		return UaasEnvData{}, fmt.Errorf("real_ip.invalid")
	}

	deviceID := r.Form.Get("device_id")

	uaasEnv := UaasEnvData{
		UserIP:     realIP,
		TestIds:    testIDs,
		SplitKey:   "uuid",
		SplitValue: deviceID,
	}
	return uaasEnv, nil
}

func (t *UaasProxy) HandlerUaas() echo.HandlerFunc {
	return func(c echo.Context) error {
		uaasEnv, err := parseForm(c.Request())
		if err != nil {
			t.unistat.uaasFailed.Inc()
			return t.respondWithError(c, err.Error())
		}
		// поход в UAAS + статистика в yasm
		t.unistat.uaasTotal.Inc()
		beforeUaas := time.Now()

		experimentInfo, err := t.QueryUaas(c.Request().Context(), &t.uaasClient, "passport_am", uaasEnv)
		duration := float64(time.Since(beforeUaas) / time.Millisecond)
		t.unistat.uaasDuration.Inc(duration)
		if err != nil {
			t.logger.Errorf("UAAS Error: %+v", err)
			if errors.Is(err, context.Canceled) {
				t.unistat.uaasCancelled.Inc()
			}
			t.unistat.uaasFailed.Inc()
			return t.respondWithError(c, "backend.uaas_failed")
		} else {
			t.unistat.uaasSuccess.Inc()
		}
		// изменение экспериментов
		formParams := make(map[string]string, len(c.Request().Form))
		for key := range c.Request().Form {
			formParams[key] = c.Request().Form.Get(key)
		}
		experimentInfo.Experiments = rewriteExperiments(
			experimentInfo.Experiments,
			// что матчить в запросе + какие флаги добавить или удалить, если запрос сматчился
			RewriteConfig,
			formParams,
		)

		document := t.pools.responsePool.Get().(SuccessResponse)
		document.Experiments = document.Experiments[:0]
		for _, experiment := range experimentInfo.Experiments {
			document.Experiments = append(
				document.Experiments,
				ResponseExperiment{
					TestID:   experiment.TestID,
					Handler:  experiment.Handler,
					Passport: experiment.Context.Passport,
				},
			)
		}
		response, err := marshal(document)
		t.pools.responsePool.Put(document)
		if err != nil {
			t.unistat.uaasFailed.Inc()
			return t.respondWithError(c, err.Error())
		}

		c.Response().Header().Set(echo.HeaderContentType, "application/json")
		return c.String(http.StatusOK, string(response))
	}
}
