package uaasproxy

import (
	"net/http"
	_ "net/http/pprof"
	"sync"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/passport/backend/uaas_proxy/internal/translations"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/unistat"
)

type UaasProxy struct {
	unistat      Unistat
	uaasClient   HTTPApiClient
	logger       log.Logger
	pools        Pools
	translations translations.TranslationsCache
}

type Pools struct {
	responsePool    sync.Pool
	flagsPool       sync.Pool
	experimentsPool sync.Pool
}

type Config struct {
	Uaas UaasConfig `json:"uaas"`
}

type UaasConfig struct {
	URL               string `json:"url"`
	UseMockedUaas     bool   `json:"mocked"`
	MockSleepDuration int    `json:"mock_sleep_duration"`
	MockJitter        int    `json:"mock_jitter"`
}

type Unistat struct {
	requestsByPath                    *unistat.SignalSet
	uaasDuration                      *unistat.SignalHgram
	uaasTotal                         *unistat.SignalDiff
	uaasSuccess                       *unistat.SignalDiff
	uaasFailed                        *unistat.SignalDiff
	uaasCancelled                     *unistat.SignalDiff
	poolFlags                         *unistat.SignalDiff
	poolExperiments                   *unistat.SignalDiff
	poolResponses                     *unistat.SignalDiff
	divTotal                          *unistat.SignalDiff
	divSuccess                        *unistat.SignalDiff
	divFailed                         *unistat.SignalDiff
	divDuration                       *unistat.SignalHgram
	divRuDarkDefaultRationale         *unistat.SignalDiff
	divEnDarkDefaultRationale         *unistat.SignalDiff
	divRuLightDefaultRationale        *unistat.SignalDiff
	divEnLightDefaultRationale        *unistat.SignalDiff
	divRuDarkDefaultBlockedRationale  *unistat.SignalDiff
	divEnDarkDefaultBlockedRationale  *unistat.SignalDiff
	divRuLightDefaultBlockedRationale *unistat.SignalDiff
	divEnLightDefaultBlockedRationale *unistat.SignalDiff
}

type Factory struct{}

func (f *Factory) NewService(config httpdaemon.ServiceConfig) (httpdaemon.Service, error) {
	var cfg Config
	if err := httpdaemon.ParseServiceConfig(config, &cfg); err != nil {
		return nil, err
	}

	var uaasClient HTTPApiClient
	if cfg.Uaas.UseMockedUaas {
		uaasClient = GetFakeUaasClient(cfg.Uaas.MockSleepDuration, cfg.Uaas.MockJitter)
	} else {
		uaasClient = GetDefaultUaasClient(cfg.Uaas.URL)
	}

	tc, err := translations.NewTranslationsCache()
	if err != nil {
		return nil, err
	}

	InitRewriteConfig()

	res := &UaasProxy{
		unistat:      InitUnistat(),
		uaasClient:   uaasClient,
		logger:       logger.Log(),
		translations: *tc,
	}
	res.InitPools()

	return res, nil
}

func (t *UaasProxy) InitPools() {
	t.pools.responsePool = sync.Pool{
		New: func() interface{} {
			t.unistat.poolResponses.Inc()
			return SuccessResponse{
				Status: "ok",
			}
		},
	}
	t.pools.flagsPool = sync.Pool{
		New: func() interface{} {
			t.unistat.poolFlags.Inc()
			return map[string]UaasResponseExperiments{}
		},
	}
	t.pools.experimentsPool = sync.Pool{
		New: func() interface{} {
			t.unistat.poolExperiments.Inc()
			return ExperimentData{}
		},
	}
}

func (t *UaasProxy) AddHandlers(e *echo.Echo) {
	e.GET(
		"/ping",
		t.HandlePing(),
	)
	e.GET("/uaas", t.HandlerUaas())
	e.GET("/1/bundle/experiments/by_device_id", t.HandlerUaas())
	e.GET("/1/bundle/experiments/by_device_id/", t.HandlerUaas())

	e.GET("/1/divcard/", t.HandlerDivcard())

	e.GET("/debug/pprof/*", echo.WrapHandler(http.DefaultServeMux))
}

func (t *UaasProxy) HandlePing() echo.HandlerFunc {
	return func(c echo.Context) error {
		return c.String(http.StatusOK, "")
	}
}
