package main

import (
	"bufio"
	"flag"
	"os"
	"os/signal"
	"runtime/pprof"
	"strconv"
	"syscall"

	"gopkg.in/ini.v1"

	"a.yandex-team.ru/passport/backend/userdata/blackbox_data_dl/internal"
	passportBlackbox "a.yandex-team.ru/passport/shared/golibs/blackbox"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

var (
	bbUIDsPerUserinfo int
	bbWorkers         int
	exportWorkers     int
	dumpingWorkers    int
	buffSize          int
	findMaxUID        bool
	dumpToStdout      bool
	gzipOutput        bool
	gzipLevel         int
	cpuProfileFile    string
	logFile           string
	configFile        string
	oldUIDsFile       string
	outputFile        string
)

func check(err error) {
	if err != nil {
		panic(err)
	}
}

func readConfig(cfg *ini.File) internal.Config {
	blackboxDst, err := strconv.ParseUint(cfg.Section("blackbox").Key("dst").String(), 10, 32)
	check(err)
	blackboxTimeout, err := cfg.Section("blackbox").Key("timeout").Duration()
	check(err)
	blackboxUIDsPerUserinfo, err := cfg.Section("blackbox").Key("uids_per_userinfo").Int()
	check(err)
	blackboxDownloadWorkers, err := cfg.Section("blackbox").Key("download_workers").Int()
	check(err)
	blackboxExportWorkers, err := cfg.Section("blackbox").Key("export_workers").Int()
	check(err)
	blackboxDumpWorkers, err := cfg.Section("blackbox").Key("dump_workers").Int()
	check(err)
	unistatListenAddr := cfg.Section("unistat").Key("listen_address").String()
	unistatListenNetwork := cfg.Section("unistat").Key("listen_network").String()
	return internal.Config{
		TVMAuthToken:            cfg.Section("tvm").Key("auth_token").String(),
		TVMSource:               cfg.Section("tvm").Key("src").String(),
		TVMAPIURL:               cfg.Section("tvm").Key("api_url").String(),
		BlackboxDestination:     uint32(blackboxDst),
		BlackboxURL:             cfg.Section("blackbox").Key("api_url").String(),
		BlackboxTimeout:         blackboxTimeout,
		BlackboxUIDsPerUserinfo: blackboxUIDsPerUserinfo,
		BlackboxDownloadWorkers: blackboxDownloadWorkers,
		BlackboxExportWorkers:   blackboxExportWorkers,
		BlackboxDumpWorkers:     blackboxDumpWorkers,
		UnistatListenAddress:    unistatListenAddr,
		UnistatListenNetwork:    unistatListenNetwork,
	}
}

func main() {
	flag.StringVar(&oldUIDsFile, "old-uids-file", "", "file with UIDs to query from blackbox")
	flag.IntVar(&gzipLevel, "gzip-level", -1, "https://golang.org/pkg/compress/flate/#DefaultCompression")
	flag.BoolVar(&findMaxUID, "find-max-uid", false, "Find newly registered accounts")
	flag.BoolVar(&dumpToStdout, "stdout", false, "Dump result to stdout")
	flag.BoolVar(&gzipOutput, "gzip", false, "GZip dumps")
	flag.StringVar(&cpuProfileFile, "cpu-profile-file", "", "Enable CPU profiling")
	flag.StringVar(&logFile, "log-file", "userdata.log", "log file")
	flag.StringVar(&configFile, "config-file", "userdata.cfg", "config file")
	flag.StringVar(&outputFile, "output-file", "result_", "File with results")
	flag.IntVar(&buffSize, "export-buffer-size", 4096, "Export buffer size for IO")
	flag.Parse()

	err := logger.Init(logger.Config{FilePath: logFile})
	check(err)
	logger.Log().Infof("Started process")

	iniCfg, err := ini.Load(configFile)
	check(err)
	cfg := readConfig(iniCfg)

	quitChan := make(chan os.Signal, 1)
	signal.Notify(quitChan, os.Interrupt, syscall.SIGTERM)

	f, err := os.Open(oldUIDsFile)
	check(err)
	defer f.Close()
	reader := bufio.NewScanner(bufio.NewReader(f))
	reader.Split(bufio.ScanLines)

	done := make(chan int)
	blackboxChannel := internal.InitBlackboxChannel()
	exportChannel := internal.InitExportChannel()

	go func() {
		<-quitChan
		logger.Log().Infof("Shutting down...")
		os.Exit(1)
	}()

	if cpuProfileFile != "" {
		f, err := os.Create(cpuProfileFile)
		check(err)
		defer f.Close()
		if err := pprof.StartCPUProfile(f); err != nil {
			logger.Log().Fatal("Could not start CPU profiling")
		}
		defer pprof.StopCPUProfile()
	}

	unistatServer := internal.GetNewUnistatServer()
	go internal.InitUnistatServer(cfg.UnistatListenAddress, cfg.UnistatListenNetwork, unistatServer)

	var maxUIDs passportBlackbox.GetMaxUIDResponse
	if findMaxUID {
		bbClient := internal.NewBlackboxClient(cfg)
		tmpMaxUIDs, err := internal.BlackboxGetMaxUID(bbClient)
		check(err)
		maxUIDs = *tmpMaxUIDs
		logger.Log().Infof("Max UID: %d, Max PDD UID: %d", maxUIDs.MaxUID, maxUIDs.MaxPDDUID)
	} else {
		logger.Log().Infof("Not looking for max UIDs")
	}

	go internal.JoinPipeline(
		unistatServer,
		cfg,
		cfg.BlackboxUIDsPerUserinfo,
		cfg.BlackboxDownloadWorkers,
		maxUIDs.MaxUID,
		maxUIDs.MaxPDDUID,
		blackboxChannel,
		exportChannel)
	go internal.ExportingChannel(
		unistatServer,
		outputFile,
		exportChannel,
		cfg.BlackboxExportWorkers,
		cfg.BlackboxDumpWorkers,
		dumpToStdout,
		gzipOutput,
		gzipLevel,
		buffSize,
		done)

	requestedUIDs := 0

	for reader.Scan() {
		text := reader.Text()
		uid, err := strconv.ParseUint(string(text[4:]), 10, 64)
		if err != nil {
			logger.Log().Fatalf("Error parsing uid from string: %+v, %+v", text, err)
			continue
		}
		requestedUIDs += 1
		blackboxChannel <- uid
	}

	logger.Log().Infof("Wanted %d UIDs in total", requestedUIDs)

	close(blackboxChannel)
	<-done
}
